#include "nodes_merger.h"

#include "../utils/geom_helpers.h"

#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <maps/libs/geolib/include/distance.h>

#include <memory>

namespace maps {
namespace wiki {
namespace topology_fixer {

namespace gl = maps::geolib3;

namespace {

inline gl::BoundingBox
pointBBox(const gl::Point2& point, double radius)
{
    return gl::BoundingBox(point,
        2.0 * (radius + gl::EPS), 2.0 * (radius + gl::EPS));
}

class NodesSearcher {
public:
    NodesSearcher(const TopologyData& data, double minNodesDistance)
        : minNodesDistance_(minNodesDistance)
    {
        auto nodeIds = data.nodeIds();
        if (nodeIds.empty()) {
            return;
        }
        searcher_.reset(new StaticNodesSearcher());
        for (auto nodeId : nodeIds) {
            auto bbox = pointBBox(data.node(nodeId).point(), minNodesDistance_);
            auto res = bboxes_.insert({nodeId, bbox});
            searcher_->insert(&res.first->second, nodeId);
        }
        searcher_->build();
    }

    boost::optional<NodeId>
    findClosestNode(const Node& node, std::function<bool(NodeId)> nodesFilter) const
    {
        ASSERT(searcher_.get());
        gl::BoundingBox searchBox = pointBBox(node.point(), minNodesDistance_);
        auto res = searcher_->find(searchBox);
        typedef std::unordered_map<NodeId, gl::BoundingBox> NodeBBoxes;
        typedef NodeBBoxes::value_type NodeBBox;
        NodeBBoxes nearNodeBBoxes;
        for (auto it = res.first; it != res.second; ++it) {
            if (it->value() != node.id() && nodesFilter(it->value())) {
                nearNodeBBoxes.insert({it->value(), it->geometry()});
            }
        }
        if (nearNodeBBoxes.empty()) {
            return boost::none;
        }
        auto it = std::min_element(nearNodeBBoxes.begin(), nearNodeBBoxes.end(),
            [&] (const NodeBBox& nb1, const NodeBBox& nb2) -> bool
            {
            return gl::distance(nb1.second.center(), node.point()) <
                gl::distance(nb2.second.center(), node.point());
            });
        ASSERT(it != nearNodeBBoxes.end());
        if (gl::distance(it->second.center(), node.point()) < minNodesDistance_) {
            return it->first;
        }
        return boost::none;
    }

private:
    typedef gl::StaticGeometrySearcher<gl::BoundingBox, NodeId> StaticNodesSearcher;

    std::unordered_map<NodeId, gl::BoundingBox> bboxes_;
    std::unique_ptr<StaticNodesSearcher> searcher_;
    double minNodesDistance_;
};

} // namespace

void
NodesMerger::operator () (TopologyData& data) const
{
    NodesSearcher searcher(data, minNodesDistance_);
    for (auto nodeId : data.nodeIds()) {
        if (!data.nodeExists(nodeId)) {
            continue;
        }
        while (true) {
            auto nearNodeId = searcher.findClosestNode(data.node(nodeId),
                [&] (NodeId id) -> bool { return data.nodeExists(id); });
            if (!nearNodeId) {
                break;
            }
            const NodeId mergeCandidateId = *nearNodeId;

            if (isNodesMergingAllowed(data, nodeId, mergeCandidateId, maxRemovableEdgeLength_)) {
                data.uniteNodes(nodeId, mergeCandidateId, maxRemovableEdgeLength_);
                INFO() << "Merged node " << mergeCandidateId << " with " << nodeId;
            } else if (isNodesMergingAllowed(data, mergeCandidateId, nodeId, maxRemovableEdgeLength_)) {
                data.uniteNodes(mergeCandidateId, nodeId, maxRemovableEdgeLength_);
                INFO() << "Merged node " << nodeId << " with " << mergeCandidateId;
                nodeId = mergeCandidateId;
            } else {
                WARN() << "Nodes " << mergeCandidateId << ", " << nodeId
                    << " can not be merged because of topology invalidation";
                break;
            }
        }
    }
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
