#include "self_overlapped_faces_remover.h"

#include "../utils/segments/segments_graph.h"
#include "../utils/geom_helpers.h"

#include <yandex/maps/wiki/common/geom_utils.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/projection.h>

#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace topology_fixer {

namespace gl = maps::geolib3;

namespace {

const double MAX_FACE_PERIMETER_TO_AREA_RATIO = 0.06; // meters / meters^2

bool
allSignificantFaceSegmentsOverlap(const TopologyData& data, FaceId faceId,
    double eps, double maxAngle)
{
    IdGenerator gen(0);
    std::map<DBIdType, gl::Segment2> faceSegments;
    for (auto edgeId : data.face(faceId).edgeIds()) {
        const auto& linestring = data.edge(edgeId).linestring();
        for (size_t i = 0; i < linestring.segmentsNumber(); ++i) {
            faceSegments.insert({gen.getId(), linestring.segmentAt(i)});
        }
    }
    bool allSegmentsOverlap = true;
    for (auto segIt = faceSegments.begin();
         allSegmentsOverlap && segIt != faceSegments.end();
         ++segIt)
    {
        if (gl::length(segIt->second) < eps) {
            continue;
        }

        bool hasOverlappingSegment = false;
        for (auto otherSegIt = faceSegments.begin();
             !hasOverlappingSegment && otherSegIt != faceSegments.end();
             ++otherSegIt)
        {
            if (otherSegIt->first == segIt->first) {
                continue;
            }
            if (gl::length(otherSegIt->second) < eps) {
                continue;
            }
            hasOverlappingSegment = hasOverlappingSegment ||
                segmentsOverlap(segIt->second, otherSegIt->second, eps, maxAngle);
        }
        allSegmentsOverlap = allSegmentsOverlap && hasOverlappingSegment;
    }
    return allSegmentsOverlap;
}

bool
faceCanBeDeleted(const TopologyData& data, FaceId faceId)
{
    bool canBeDeleted = true;
    auto faceMasters = data.face(faceId).masterIds();
    for (auto masterIt = faceMasters.begin();
         canBeDeleted && masterIt != faceMasters.end();
         ++masterIt)
    {
        const Master& master = data.master(*masterIt);
        const FaceRelationType faceRelType = master.relationType(faceId);
        const auto& ftAttrs = master.ftAttributes();
        canBeDeleted = canBeDeleted &&
            ((faceRelType == FaceRelationType::Exterior && (
                  (ftAttrs && ftAttrs->canBeDeleted()) ||
                   master.faceIds(FaceRelationType::Exterior).size() > 1))
            || faceRelType == FaceRelationType::Interior);
    }
    return canBeDeleted;
}

bool
faceIsDegenerate(const TopologyData& data, FaceId faceId,
    double eps, double maxAngle, double minAdFaceArea)
{
    try {
        auto points = data.faceVertices(faceId);

        double perimeter = topology_fixer::perimeter(points);
        if (data.srid() == SRID::Mercator) {
            perimeter = geolib3::toMeters(perimeter, points.front());
        }

        if (perimeter < eps) {
            INFO() << "Face " << faceId
                << " is degenerate because perimeter is too small: " << perimeter;
            return true;
        }

        double area = std::fabs(topology_fixer::signedArea(points));
        if (data.srid() == SRID::Mercator) {
            area = geolib3::toSquareMeters(area, points.front());
        }

        if (data.ftGroup()->id() == TopologyGroupId::Ad && area < minAdFaceArea) {
            INFO() << "Face " << faceId << " has too small area for ad group, area "
                << area;
            return true;
        }
        if (data.ftGroup()->id() == TopologyGroupId::Ad && perimeter / area > MAX_FACE_PERIMETER_TO_AREA_RATIO) {
            INFO() << "Face " << faceId << " has too big perimeter to area ratio for ad group: "
                << perimeter / area;
            return true;
        }
        if (allSignificantFaceSegmentsOverlap(data, faceId, eps, maxAngle) &&
            area < eps * perimeter / 2.0)
        {
            INFO() << "Face " << faceId
                << " is degenerate because it is self-overlapping and area/perimeter ratio is too small: "
                << area / perimeter
                << ", area: " << area << ", perimeter: " << perimeter;
            return true;
        }
    } catch (const std::exception& e) {
        WARN() << "Face " << faceId << " geometry is invalid: " << e.what();
    } catch (...) {
        WARN() << "Face " << faceId << " geometry is invalid: unknown error";
    }
    return false;
}

} // namespace

void
SelfOverlappedFacesRemover::operator () (TopologyData& data) const
{
    IdSet deletedFaceIds;
    for (auto faceId : data.faceIds()) {
        if (deletedFaceIds.count(faceId)) {
            continue;
        }
        if (!faceIsDegenerate(data, faceId, eps_, maxAngle_, minAdFaceArea_)) {
            continue;
        }
        if (!faceCanBeDeleted(data, faceId)) {
            WARN() << "Face " << faceId
                << " can not be removed because master attributes prevent deletion";
            continue;
        }
        auto faceMasters = data.face(faceId).masterIds();
        data.removeFace(faceId, DeletionMode::Cascade);
        deletedFaceIds.insert(faceId);
        INFO() << "Face " << faceId << " removed";
        for (auto masterId : faceMasters) {
            const auto& faceRels = data.master(masterId).faceRels();
            bool hasExternalFaces = std::count_if(
                faceRels.begin(), faceRels.end(),
                [] (const FaceRelations::value_type& rel) -> bool
                {
                    return rel.second == FaceRelationType::Exterior;
                }) > 0;
            if (!hasExternalFaces) {
                IdSet holeIds;
                std::transform(
                    faceRels.begin(), faceRels.end(), std::inserter(holeIds, holeIds.end()),
                    [] (const FaceRelations::value_type& r) -> DBIdType { return r.first; });
                data.removeMaster(masterId, DeletionMode::Cascade);
                INFO() << "Master " << masterId << " has no faces, removed";
                deletedFaceIds.insert(holeIds.begin(), holeIds.end());
            }
        }
    }
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
