#pragma once

#include <maps/libs/xml/include/xml.h>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace config {

class EdgeGeometry {
public:
    explicit EdgeGeometry(const xml3::Node& node)
        : maxEdgeVertices_(node.node("max-edge-vertices").value<size_t>())
        , minSegmentLength_(node.node("min-segment-length").value<double>())
    {}

    size_t maxEdgeVertices() const { return maxEdgeVertices_; }
    double minSegmentLength() const { return minSegmentLength_; }

private:
    size_t maxEdgeVertices_;
    double minSegmentLength_;
};


class NodeMerging {
public:
    explicit NodeMerging(const xml3::Node& node)
        : minNodesDistance_(node.node("min-nodes-distance").value<double>())
        , maxRemovableEdgeLength_(node.node("max-removable-edge-length").value<double>())
    {}

    double minNodesDistance() const { return minNodesDistance_; }
    double maxRemovableEdgeLength() const { return maxRemovableEdgeLength_; }

private:
    double minNodesDistance_;
    double maxRemovableEdgeLength_;
};

class EdgeOverlap {
public:
    explicit EdgeOverlap(const xml3::Node& node)
        : tolerance_(node.node("tolerance").value<double>())
        , maxAngleBetweenSegments_(
            node.node("max-angle-between-segments").value<double>())
    {}

    double tolerance() const { return tolerance_; }
    double maxAngleBetweenSegments() const { return maxAngleBetweenSegments_; }

private:
    double tolerance_;
    double maxAngleBetweenSegments_;
};


class EdgeIntersection {
public:
    explicit EdgeIntersection(const xml3::Node& node)
        : tolerance_(node.node("tolerance").value<double>())
    {}

    double tolerance() const { return tolerance_; }

private:
    double tolerance_;
};


class DegenerateFace {
public:
    explicit DegenerateFace(const xml3::Node& node)
        : maxFaceWidth_(node.node("max-face-width").value<double>())
        , maxAngleBetweenSegments_(
            node.node("max-angle-between-segments").value<double>())
        , minAdFaceArea_(
            node.node("min-ad-face-area").value<double>())
    {}

    double maxFaceWidth() const { return maxFaceWidth_; }
    double maxAngleBetweenSegments() const { return maxAngleBetweenSegments_; }
    double minAdFaceArea() const { return minAdFaceArea_; }

private:
    double maxFaceWidth_;
    double maxAngleBetweenSegments_;
    double minAdFaceArea_;
};

class FaceMerging {
public:
    explicit FaceMerging(const xml3::Node& node)
        : tolerance_(node.node("tolerance").value<double>())
        , maxAngleBetweenSegments_(
              node.node("max-angle-between-segments").value<double>())
        , maxTranslationFactor_(
              node.node("max-translation-factor").value<double>())
        , maxAPRatioFactor_(
              node.node("max-ap-ratio-factor").value<double>())
    {}

    double tolerance() const { return tolerance_; }
    double maxAngleBetweenSegments() const { return maxAngleBetweenSegments_; }
    double maxTranslationFactor() const { return maxTranslationFactor_; }
    double maxAPRatioFactor() const { return maxAPRatioFactor_; }

private:
    double tolerance_;
    double maxAngleBetweenSegments_;
    double maxTranslationFactor_;
    double maxAPRatioFactor_;
};

class FaceGap {
public:
    explicit FaceGap(const xml3::Node& node)
        : maxGapWidth_(node.node("max-gap-width").value<double>())
        , maxGapLengthFactor_(
              node.node("max-gap-length-factor").value<double>())
    {}

    double maxGapWidth() const { return maxGapWidth_; }
    double maxGapLengthFactor() const { return maxGapLengthFactor_; }

private:
    double maxGapWidth_;
    double maxGapLengthFactor_;
};


class FtMerging {
public:
    explicit FtMerging(const xml3::Node& node)
        : maxMasterFaces_(node.node("max-master-faces").value<size_t>())
        , maxFaceEdges_(
              node.node("max-face-edges").value<size_t>())
        , maxFaceBBoxSize_(
              node.node("max-face-bbox-size").value<double>())
    {}

    size_t maxMasterFaces() const { return maxMasterFaces_; }
    size_t maxFaceEdges() const { return maxFaceEdges_; }
    double maxFaceBBoxSize() const { return maxFaceBBoxSize_; }

private:
    size_t maxMasterFaces_;
    size_t maxFaceEdges_;
    double maxFaceBBoxSize_;
};


class Restrictions {
public:

    explicit Restrictions(const xml3::Node& restrictionsNode)
        : edgeGeometry_(restrictionsNode.node("edge-geometry"))
        , nodeMerging_(restrictionsNode.node("node-merging"))
        , edgeOverlap_(restrictionsNode.node("edge-overlap"))
        , edgeIntersection_(restrictionsNode.node("edge-intersection"))
        , degenerateFace_(restrictionsNode.node("degenerate-face"))
        , faceMerging_(restrictionsNode.node("face-merging"))
        , faceGap_(restrictionsNode.node("face-gap"))
        , ftMerging_(restrictionsNode.node("ft-merging"))
    {}

    const EdgeGeometry& edgeGeometry() const { return edgeGeometry_; }
    const NodeMerging& nodeMerging() const { return nodeMerging_; }
    const EdgeOverlap& edgeOverlap() const { return edgeOverlap_; }
    const EdgeIntersection& edgeIntersection() const { return edgeIntersection_; }
    const DegenerateFace& degenerateFace() const { return degenerateFace_; }
    const FaceMerging& faceMerging() const { return faceMerging_; }
    const FaceGap& faceGap() const { return faceGap_; }
    const FtMerging& ftMerging() const { return ftMerging_; }

private:
    EdgeGeometry edgeGeometry_;
    NodeMerging nodeMerging_;
    EdgeOverlap edgeOverlap_;
    EdgeIntersection edgeIntersection_;
    DegenerateFace degenerateFace_;
    FaceMerging faceMerging_;
    FaceGap faceGap_;
    FtMerging ftMerging_;
};

} // namespace config
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
