#include "topology_data.h"

#include "db_strings.h"
#include "utils/geom_helpers.h"

#include <boost/lexical_cast.hpp>

#include <iostream>
#include <string>
#include <sstream>
#include <array>
#include <type_traits>

namespace geolib = maps::geolib3;

namespace maps {
namespace wiki {
namespace topology_fixer {

namespace {

template <class MapReference, class Reference>
Reference
value(
    MapReference data,
    const typename std::remove_reference<MapReference>::type::key_type& key,
    const std::string& objectTypeName)
{
    try {
        return data.at(key);
    } catch (const std::out_of_range&) {
        throw MissingObjectException() << objectTypeName << " with id "
            << key << " not found";
    }
}

const DispClassId MIN_DISP_CLASS_ID = 1;
const DispClassId MAX_DISP_CLASS_ID = 10;

const SearchClassId MIN_SEARCH_CLASS_ID = 1;
const SearchClassId MAX_SEARCH_CLASS_ID = 10;

} // namespace

Edge::Edge(
        DBIdType id,
        geolib3::Polyline2 linestring,
        DBIdType fnodeId,
        DBIdType tnodeId,
        ZLevelType fZlev,
        ZLevelType tZlev)
    : id_(id)
    , fnodeId_(fnodeId)
    , tnodeId_(tnodeId)
    , fZlev_(fZlev)
    , tZlev_(tZlev)
    , linestring_(std::move(linestring))
    , boundingBox_(linestring_.boundingBox())
{}

void Edge::setNewGeometry(geolib3::Polyline2 geom)
{
    linestring_ = std::move(geom);
    boundingBox_ = linestring_.boundingBox();
}

AttributesMap Edge::getAttrs() const
{
    return {
        {FIELD_EDGE_ID, std::to_string(id_)},
        {FIELD_T_NODE_ID, std::to_string(tnodeId_)},
        {FIELD_F_NODE_ID, std::to_string(fnodeId_)},
        {FIELD_T_ZLEV, std::to_string(fZlev_)},
        {FIELD_F_ZLEV, std::to_string(tZlev_)}
    };
}

FtAttributes::FtAttributes(
        FtTypeId ftTypeId,
        DispClassId dispClassId,
        boost::optional<SearchClassId> searchClassId,
        boost::optional<MasterId> parentId,
        const boost::optional<std::string>& isocode,
        const boost::optional<std::string>& subcode)
    : ftTypeId_(ftTypeId)
    , dispClassId_(dispClassId)
    , searchClassId_(searchClassId)
    , hasName_(false)
    , isAddrAssociatedWith_(false)
    , hasChildren_(false)
    , centerId_(boost::none)
    , parentId_(parentId)
    , adId_(boost::none)
    , isocode_(isocode)
    , subcode_(subcode)
{
    REQUIRE(
        MIN_DISP_CLASS_ID <= dispClassId_ && dispClassId_ <= MAX_DISP_CLASS_ID,
        "Invalid disp_class value " << dispClassId_);
    if (searchClassId_) {
        REQUIRE(
            MIN_SEARCH_CLASS_ID <= *searchClassId_ && *searchClassId_ <= MAX_SEARCH_CLASS_ID,
            "Invalid search_class value " << *searchClassId_);
    }
}

bool
FtAttributes::canBeDeleted() const
{
    if (hasName_) {
        INFO() << "has name";
    }
    if (isAddrAssociatedWith_) {
        INFO() << "addr associated";
    }
    if (hasChildren_) {
        INFO() << "has children";
    }
    return !hasName_ && !isAddrAssociatedWith_ && !hasChildren_;
}

bool
FtAttributes::canBeMergedWith(const FtAttributes& toLeave) const
{
    return canBeDeleted() && !toLeave.hasName() &&
        ftTypeId_ == toLeave.ftTypeId() &&
        dispClassId_ == toLeave.dispClassId() &&
        searchClassId_ == toLeave.searchClassId() &&
        isocode_ == toLeave.isocode() && subcode_ == toLeave.subcode();
}

boost::iterator_range<FaceRelations::const_iterator>
Master::faceRels() const
{
    return boost::make_iterator_range(faceRels_.cbegin(), faceRels_.cend());
}

boost::iterator_range<FaceRelations::iterator>
Master::faceRels()
{
    return boost::make_iterator_range(faceRels_.begin(), faceRels_.end());
}

IdSet
Master::faceIds() const
{
    IdSet faceIds;
    std::transform(
        faceRels_.begin(), faceRels_.end(),
        std::inserter(faceIds, faceIds.end()),
        [] (const std::pair<FaceId, FaceRelationType>& faceRel) -> DBIdType
        {
            return faceRel.first;
        });
    return faceIds;
}

IdSet
Master::faceIds(FaceRelationType relationType) const
{
    IdSet faceIds;
    for (const auto& faceRel : faceRels_) {
        if (faceRel.second == relationType) {
            faceIds.insert(faceRel.first);
        }
    }
    return faceIds;
}

FaceRelationType
Master::relationType(DBIdType faceId) const
{
    return value<const FaceRelations&, FaceRelationType>(faceRels_, faceId, "Master relation to face");
}

// TopologyData

std::unique_ptr<TopologyData> TopologyData::clone() const
{
    std::list<NodeData> nodes;
    std::list<EdgeData> edges;
    std::list<FaceEdgeData> faceEdgeRels;
    std::list<EdgeMasterData> masterEdgeRels;
    std::list<FaceMasterData> masterFaceRels;

    for (const auto& nodeData : nodes_) {
        nodes.push_back({nodeData.first, nodeData.second.point()});
    }
    for (const auto& edgeData : edges_) {
        const Edge& e = edgeData.second;
        edges.push_back({edgeData.first,
            e.linestring(), e.fnodeId(), e.tnodeId(), e.fZlev(), e.tZlev()});
    }
    for (const auto& faceData : faces_) {
        for (auto edgeId : faceData.second.edgeIds()) {
            faceEdgeRels.push_back({faceData.first, edgeId});
        }
    }
    for (const auto& masterData : masters_) {
        for (const auto& faceRel : masterData.second.faceRels()) {
            masterFaceRels.push_back({masterData.first, faceRel.first, faceRel.second});
        }
        for (auto edgeId : masterData.second.edgeIds()) {
            masterEdgeRels.push_back({masterData.first, edgeId});
        }
    }
    std::unique_ptr<TopologyData> result(
        new TopologyData(ftGroup_,
            nodes, edges, faceEdgeRels, masterEdgeRels, masterFaceRels,
            idGen_, srid_));
    for (const auto& masterData : masters_) {
        if (masterData.second.ftAttributes()) {
            result->setFtAttributes(masterData.first, *masterData.second.ftAttributes());
        }
    }
    return result;
}

Node& TopologyData::node(DBIdType id)
{
    return value<Nodes&, Node&>(nodes_, id, "Node");
}

Edge& TopologyData::edge(DBIdType id)
{
    return value<Edges&, Edge&>(edges_, id, "Edge");
}

Face& TopologyData::face(DBIdType id)
{
    return value<Faces&, Face&>(faces_, id, "Face");
}

Master& TopologyData::master(DBIdType id)
{
    return value<Masters&, Master&>(masters_, id, "Master");
}

const Node& TopologyData::node(DBIdType id) const
{
    return value<const Nodes&, const Node&>(nodes_, id, "Node");
}

const Edge& TopologyData::edge(DBIdType id) const
{
    return value<const Edges&, const Edge&>(edges_, id, "Edge");
}

const Face& TopologyData::face(DBIdType id) const
{
    return value<const Faces&, const Face&>(faces_, id, "Face");
}

const Master& TopologyData::master(DBIdType id) const
{
    return value<const Masters&, const Master&>(masters_, id, "Master");
}

boost::iterator_range<Nodes::iterator> TopologyData::nodes() {
    return boost::make_iterator_range(nodes_.begin(), nodes_.end());
}

boost::iterator_range<Edges::iterator> TopologyData::edges() {
    return boost::make_iterator_range(edges_.begin(), edges_.end());
}

boost::iterator_range<Faces::iterator> TopologyData::faces() {
    return boost::make_iterator_range(faces_.begin(), faces_.end());
}

boost::iterator_range<Masters::iterator> TopologyData::masters() {
    return boost::make_iterator_range(masters_.begin(), masters_.end());
}

boost::iterator_range<Nodes::const_iterator> TopologyData::nodes() const {
    return boost::make_iterator_range(nodes_.cbegin(), nodes_.cend());
}

boost::iterator_range<Edges::const_iterator> TopologyData::edges() const {
    return boost::make_iterator_range(edges_.cbegin(), edges_.cend());
}

boost::iterator_range<Faces::const_iterator> TopologyData::faces() const {
    return boost::make_iterator_range(faces_.cbegin(), faces_.cend());
}

boost::iterator_range<Masters::const_iterator> TopologyData::masters() const {
    return boost::make_iterator_range(masters_.cbegin(), masters_.cend());
}

bool TopologyData::isNodeLinear(NodeId nodeId) const
{
    const Node& node = this->node(nodeId);
    for (auto edgeId : node.edgeIds()) {
        const Edge& edge = this->edge(edgeId);
        if (!edge.isLinear()) {
            return false;
        }
    }
    return true;
}

bool TopologyData::isNodeContour(NodeId nodeId) const
{
    const Node& node = this->node(nodeId);
    for (auto edgeId : node.edgeIds()) {
        const Edge& edge = this->edge(edgeId);
        if (!edge.isContour()) {
            return false;
        }
    }
    return true;
}

std::vector<geolib::Point2> TopologyData::faceVertices(DBIdType faceId) const
{
    const Face& face = this->face(faceId);
    const auto& edgeIds = face.edgeIds();

    if (edgeIds.empty()) {
        throw InvalidFaceException() << "Face " << face.id() << " has no edges";
    }
    if (edgeIds.size() == 1) {
        const Edge& edge = this->edge(*edgeIds.begin());
        if (edge.fnodeId() != edge.tnodeId()) {
            throw InvalidFaceException() << "Face " << face.id() << " is not closed";
        }
        return edge.linestring().points();
    }

    PlainEdgesData edgesData;

    for (auto edgeId : edgeIds) {
        const auto& edge = this->edge(edgeId);
        const auto fnodeId = edge.fnodeId();
        const auto tnodeId = edge.tnodeId();
        edgesData.emplace(edgeId, PlainEdgeData{fnodeId, tnodeId, edge.linestring().points()});
    }

    return buildPolygon(faceId, edgesData);
}

void TopologyData::setFNode(DBIdType edgeId, DBIdType nodeId)
{
    Edge& edge = this->edge(edgeId);
    if (edge.fnodeId() == nodeId) {
        return;
    }
    Node& oldFNode = this->node(edge.fnodeId());
    Node& newFNode = this->node(nodeId);

    if (edge.fnodeId() != edge.tnodeId()) {
        oldFNode.edgeIds_.erase(edgeId);
    }

    auto points = edge.linestring().points();
    points.front() = newFNode.point();
    edge.setNewGeometry(geolib3::Polyline2(std::move(points)));

    newFNode.edgeIds_.insert(edgeId);
    edge.fnodeId_ = nodeId;
}

void TopologyData::setTNode(DBIdType edgeId, DBIdType nodeId)
{
    Edge& edge = this->edge(edgeId);
    if (edge.tnodeId() == nodeId) {
        return;
    }
    Node& oldTNode = this->node(edge.tnodeId());
    Node& newTNode = this->node(nodeId);

    if (edge.fnodeId() != edge.tnodeId()) {
        oldTNode.edgeIds_.erase(edgeId);
    }

    auto points = edge.linestring().points();
    points.back() = newTNode.point();
    edge.setNewGeometry(geolib3::Polyline2(std::move(points)));

    newTNode.edgeIds_.insert(edgeId);
    edge.tnodeId_ = nodeId;
}

void TopologyData::addFaceEdge(DBIdType faceId, DBIdType edgeId) {
    Face& face = this->face(faceId);
    Edge& edge = this->edge(edgeId);
    face.edgeIds_.insert(edge.id());
    edge.faceIds_.insert(face.id());
}

void TopologyData::addMasterFaceRel(
    DBIdType masterId, DBIdType faceId, FaceRelationType isInterior)
{
    Master& master = this->master(masterId);
    Face& face = this->face(faceId);
    master.faceRels_.insert({face.id(), isInterior});
    face.masterIds_.insert(master.id());
}

void TopologyData::addMasterEdgeRel(DBIdType masterId, DBIdType edgeId)
{
    Master& master = this->master(masterId);
    Edge& edge = this->edge(edgeId);
    master.edgeIds_.insert(edge.id());
    edge.masterIds_.insert(master.id());
}

void TopologyData::setFtAttributes(MasterId ftId, const FtAttributes& attributes)
{
    REQUIRE(ftGroup()->id() != TopologyGroupId::Ad,
        "Topology group " << ftGroup()->name() << " is no ft group");
    auto it = masters_.find(ftId);
    if (it == masters_.end()) {
        WARN() << "Master " << ftId << " is not found, attributes skipped";
        return;
    }
    Master& master = it->second;
    master.ftAttributes_ = attributes;
}

namespace {
template<typename ValueType>
DBIdType mapKey(const std::pair<DBIdType, ValueType>& pair) {
    return pair.first;
}
} // namespace

std::vector<DBIdType> TopologyData::nodeIds() const {
    std::vector<DBIdType> result;
    result.reserve(nodes_.size());
    std::transform(nodes_.begin(), nodes_.end(), std::back_inserter(result), mapKey<Node>);
    return result;
}

std::vector<DBIdType> TopologyData::edgeIds() const {
    std::vector<DBIdType> result;
    result.reserve(edges_.size());
    std::transform(edges_.begin(), edges_.end(), std::back_inserter(result), mapKey<Edge>);
    return result;
}

std::vector<DBIdType> TopologyData::faceIds() const {
    std::vector<DBIdType> result;
    result.reserve(faces_.size());
    std::transform(faces_.begin(), faces_.end(), std::back_inserter(result), mapKey<Face>);
    return result;
}

bool TopologyData::canRemoveNode(const Node& node) const {
    return node.edgeIds().empty();
}

void TopologyData::removeNode(DBIdType id) {
    REQUIRE(canRemoveNode(node(id)),
        "Node " << id << " must be removed but is not isolated");
    nodes_.erase(id);
}

void TopologyData::removeEdge(DBIdType id, DeletionMode mode)
{
    Edge& edge = this->edge(id);
    Node& fnode = node(edge.fnodeId());
    Node& tnode = node(edge.tnodeId());
    size_t fedgeSize = fnode.edgeIds().size();
    size_t tedgeSize = tnode.edgeIds().size();

    REQUIRE(edge.faceIds().empty(),
        "Edge " << id << " can not be deleted"
        " because it is still referenced by face: " << *edge.faceIds().begin());
    REQUIRE(edge.masterIds().empty(),
        "Edge " << id << " can not be deleted"
        " because it is still referenced by master: " << *edge.masterIds().begin());

    if (fedgeSize == 1 && mode == DeletionMode::Cascade) {
        nodes_.erase(edge.fnodeId());
    }
    else {
        fnode.edgeIds_.erase(edge.id());
    }

    if (tedgeSize == 1 && mode == DeletionMode::Cascade) {
        nodes_.erase(edge.tnodeId());
    }
    else {
        tnode.edgeIds_.erase(edge.id());
    }

    edges_.erase(id);
}

void TopologyData::removeFace(DBIdType id, DeletionMode mode)
{
    Face& face = this->face(id);

    REQUIRE(mode != DeletionMode::Restrict || face.masterIds().empty(),
        "Face " << id << " can not be deleted "
        "because it is still referenced by master " << *face.masterIds().begin());
    REQUIRE(mode != DeletionMode::Restrict || face.edgeIds().empty(),
        "Face " << id << " can not be deleted "
        "because it is still references edge " << *face.edgeIds().begin());

    auto edgeIds = face.edgeIds();
    for (DBIdType edgeId : edgeIds) {
        Edge& edge = this->edge(edgeId);
        edge.faceIds_.erase(id);
        if (mode == DeletionMode::Cascade &&
            edge.faceIds().empty() && edge.masterIds().empty())
        {
            removeEdge(edgeId, DeletionMode::Cascade);
        }
    }

    for (DBIdType masterId : face.masterIds()) {
        master(masterId).faceRels_.erase(face.id());
    }

    faces_.erase(id);
}

void TopologyData::removeMaster(DBIdType id, DeletionMode mode)
{
    Master& master = this->master(id);
    REQUIRE(!master.ftAttributes_ || master.ftAttributes_->canBeDeleted(),
        "Master " << id << " is not removable because of feature attrs");
    REQUIRE(mode != DeletionMode::Restrict || master.faceRels().empty(),
        "Master " << id << " can not be deleted "
        "because it is still referenced by face " << master.faceRels().begin()->first);
    REQUIRE(mode != DeletionMode::Restrict || master.edgeIds().empty(),
        "Master " << id << " can not be deleted "
        "because it is still referenced by edge " << *master.edgeIds().begin());
    // DeletionMode::Cascade
    auto faceIds = master.faceIds();
    for (auto faceId : faceIds) {
        Face& face = this->face(faceId);
        face.masterIds_.erase(id);
        if (face.masterIds().empty()) {
            removeFace(faceId, DeletionMode::Cascade);
        }
    }
    auto edgeIds = master.edgeIds();
    for (auto edgeId : edgeIds) {
        Edge& edge = this->edge(edgeId);
        edge.masterIds_.erase(id);
        if (edge.masterIds().empty() && edge.faceIds().empty()) {
            removeEdge(edgeId, DeletionMode::Cascade);
        }
    }
    if (master.ftAttributes_) {
        deletedFtAttributes_.insert({id, std::move(*master.ftAttributes_)});
    }
    masters_.erase(id);
}

void TopologyData::removeFaceEdgeRel(DBIdType faceId, DBIdType edgeId)
{
    auto& face = this->face(faceId);
    auto& edge = this->edge(edgeId);
    face.edgeIds_.erase(edgeId);
    edge.faceIds_.erase(faceId);
}

void TopologyData::removeMasterFaceRel(DBIdType masterId, DBIdType faceId)
{
    auto& master = this->master(masterId);
    auto& face = this->face(faceId);

    master.faceRels_.erase(faceId);
    face.masterIds_.erase(masterId);
}

void TopologyData::removeMasterEdgeRel(DBIdType masterId, DBIdType edgeId)
{
    auto& master = this->master(masterId);
    auto& edge = this->edge(edgeId);

    master.edgeIds_.erase(edgeId);
    edge.masterIds_.erase(masterId);
}

DBIdType
TopologyData::addNode(const geolib3::Point2& point)
{
    return addNodeInternal(getId(), point);
}

void
TopologyData::addExistingNode(DBIdType id, const geolib3::Point2& point)
{
    addNodeInternal(id, point);
}

DBIdType TopologyData::addEdge(
    geolib::Polyline2 linestring,
    DBIdType fnodeId,
    DBIdType tnodeId,
    ZLevelType fZlev,
    ZLevelType tZlev)
{
    return addEdgeInternal(getId(), std::move(linestring), fnodeId, tnodeId, fZlev, tZlev);
}

void
TopologyData::addExistingEdge(
    DBIdType id,
    geolib::Polyline2 linestring,
    DBIdType fnodeId,
    DBIdType tnodeId,
    ZLevelType fZlev,
    ZLevelType tZlev)
{
    addEdgeInternal(id, std::move(linestring), fnodeId, tnodeId, fZlev, tZlev);
}

DBIdType TopologyData::addNodeInternal(
    DBIdType id,
    const geolib3::Point2& point)
{
    nodes_.insert({id, Node(id, point)});
    return id;
}

DBIdType TopologyData::addEdgeInternal(
    DBIdType edgeId,
    geolib::Polyline2 linestring,
    DBIdType fnodeId,
    DBIdType tnodeId,
    ZLevelType fZlev,
    ZLevelType tZlev,
    CheckMisplacedNodes checkMisplacedNodes)
{
    auto& fnode = node(fnodeId);
    if (checkMisplacedNodes == CheckMisplacedNodes::Yes) {
        REQUIRE(fnode.point() == linestring.points().front(),
            "Edge " << edgeId << " geometry start point is not at fnode " << fnodeId << " pos");
    }
    auto& tnode = node(tnodeId);
    if (checkMisplacedNodes == CheckMisplacedNodes::Yes) {
        REQUIRE(tnode.point() == linestring.points().back(),
            "Edge " << edgeId << " geometry end point is not at tnode " << tnodeId << " pos");
    }
    fnode.edgeIds_.insert(edgeId);
    tnode.edgeIds_.insert(edgeId);
    REQUIRE(edges_.emplace(
        edgeId,
        Edge(
            edgeId,
            std::move(linestring),
            fnodeId, tnodeId,
            fZlev, tZlev)
        ).second,
        "Edge " << edgeId << " already exists");
    return edgeId;
}

DBIdType TopologyData::addFace()
{
    DBIdType faceId = getId();
    faces_.insert({faceId, Face(faceId)});
    return faceId;
}

void TopologyData::addExistingFace(DBIdType id)
{
    faces_.insert({id, Face(id)});
}

DBIdType TopologyData::addMaster()
{
    DBIdType masterId = getId();
    masters_.insert({masterId, Master(masterId)});
    return masterId;
}

DBIdType TopologyData::addMaster(const FtAttributes& ftAttrs)
{
    DBIdType masterId = getId();
    masters_.insert({masterId, Master(masterId, ftAttrs)});
    return masterId;
}

void TopologyData::addExistingMaster(DBIdType masterId)
{
    masters_.insert({masterId, Master(masterId)});
}

void
TopologyData::addExistingMaster(DBIdType masterId, const FtAttributes& ftAttrs)
{
    masters_.insert({masterId, Master(masterId, ftAttrs)});
}

void TopologyData::uniteNodes(DBIdType toLeaveId, DBIdType toRemoveId,
    double maxRemovableEdgeLength)
{
    if (toLeaveId == toRemoveId) {
        return;
    }

    Node& toLeave = node(toLeaveId);
    Node& toRemove = node(toRemoveId);

    IdSet removedEdgeIds;
    auto movedEdgeIds = toRemove.edgeIds();
    for (auto edgeId : movedEdgeIds) {
        Edge& edge = this->edge(edgeId);
        const NodeId oldFnodeId = edge.fnodeId();
        const NodeId oldTnodeId = edge.tnodeId();
        geolib::PointsVector points = edge.linestring().points();
        if (edge.fnodeId_ == toRemoveId) {
            edge.fnodeId_ = toLeaveId;
            points.front() = toLeave.point();
        }
        if (edge.tnodeId_ == toRemoveId) {
            edge.tnodeId_ = toLeaveId;
            points.back() = toLeave.point();
        }
        if (oldFnodeId != oldTnodeId && edge.fnodeId() == edge.tnodeId()) {
            const double length = geolib::length(edge.linestring());
            if (ftGroup_->topologyType() == TopologyType::Contour) {
                REQUIRE(length < maxRemovableEdgeLength,
                    "Edge " << edgeId << " length is too big for edge to be deleted"
                    << ", nodes " << toRemoveId << ", " << toLeaveId << " merging is not possible");
            }
            if (length < maxRemovableEdgeLength) {
                removedEdgeIds.insert(edgeId);
                continue;
            }
        }
        edge.setNewGeometry(geolib3::Polyline2(std::move(points)));
        toLeave.edgeIds_.insert(edge.id());
    }
    for (auto edgeId : removedEdgeIds) {
        const Edge& edge = this->edge(edgeId);
        auto masterIds = edge.masterIds();
        for (auto masterId : masterIds) {
            removeMasterEdgeRel(masterId, edgeId);
        }
        auto faceIds = edge.faceIds();
        for (auto faceId : faceIds) {
            removeFaceEdgeRel(faceId, edgeId);
        }
        removeEdge(edgeId, DeletionMode::Restrict);
    }

    nodes_.erase(toRemoveId);
}

void TopologyData::copyFaceRelations(const Edge& source, DBIdType target)
{
    edge(target).faceIds_ = source.faceIds_;
    for (auto faceId : source.faceIds()) {
        face(faceId).edgeIds_.insert(target);
    }
}

DBIdType TopologyData::getId()
{
    return idGen_->getId();
}

void TopologyData::replaceEdge(DBIdType edgeId, const IdSet& newEdgeIds)
{
    const Edge& edge = this->edge(edgeId);
    auto faceIds = edge.faceIds();
    for (auto faceId : faceIds) {
        for (auto newEdgeId : newEdgeIds) {
            addFaceEdge(faceId, newEdgeId);
        }
        removeFaceEdgeRel(faceId, edgeId);
    }
    auto masterIds = edge.masterIds();
    for (auto masterId : masterIds) {
        for (auto newEdgeId : newEdgeIds) {
            addMasterEdgeRel(masterId, newEdgeId);
        }
        removeMasterEdgeRel(masterId, edgeId);
    }
    ASSERT(edge.faceIds().empty());
    ASSERT(edge.masterIds().empty());
    removeEdge(edgeId, DeletionMode::Cascade);
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
