#include "topology_group.h"

#include "db_strings.h"
#include "common.h"

#include <boost/lexical_cast.hpp>

#include <stdexcept>

namespace maps {
namespace wiki {
namespace topology_fixer {

const std::string& TopologyGroup::masterTable() const
{
    return id_ == TopologyGroupId::Ad ? TABLE_AD : TABLE_FT;
}

std::string TopologyGroup::sqlFtTypeFilter(const std::string& ftTypeIdField) const
{
    if (ftTypes_.empty()) {
        return "";
    }
    if (ftTypes_.size() == 1) {
        return "(" + ftTypeIdField
            + " = " + boost::lexical_cast<std::string>(*ftTypes_.begin()) + ")";
    }
    return "(" + ftTypeIdField
        + " IN (" + toString(ftTypes_) + "))";
}

TopologyGroupsRegistry::TopologyGroupsRegistry()
{
    std::map<TopologyGroupId, TopologyGroup::FtTypes> ftTypes = {
        {TopologyGroupId::Ad, {}},
        {TopologyGroupId::Vegetation, {401, 402, 403, 404}},
        {TopologyGroupId::HydroContour, {501, 502, 503, 504, 505, 506, 507, 508, 509, 510, 551, 552, 553, 554, 555, 556, 557, 558}},
        {TopologyGroupId::HydroLn, {551, 552, 553, 554, 555, 556, 557, 558}},
        {TopologyGroupId::Urban, {101, 102, 110, 111, 112, 113, 114, 171, 172, 173, 191, 201, 202, 221, 223, 224, 225, 226}},
        {TopologyGroupId::UrbanRoadnet, {241, 242, 243, 251, 252}},
        {TopologyGroupId::Relief, {301, 302, 303, 304}},
        {TopologyGroupId::Landmark, {1, 2}},
        {TopologyGroupId::TransportRailway, {611, 612, 613, 614, 615, 616, 617, 618, 619, 620}}
    };
    std::map<TopologyGroupId, std::string> names = {
        {TopologyGroupId::Ad, AD},
        {TopologyGroupId::Vegetation, VEGETATION},
        {TopologyGroupId::HydroContour, CONTOUR_HYDROGRAPHY},
        {TopologyGroupId::HydroLn, LINEAR_HYDROGRAPHY},
        {TopologyGroupId::Urban, URBAN},
        {TopologyGroupId::UrbanRoadnet, URBAN_ROADNET},
        {TopologyGroupId::Relief, RELIEF},
        {TopologyGroupId::Landmark, LANDMARK},
        {TopologyGroupId::TransportRailway, TRANSPORT_RAILWAY}
    };

    for (const auto& groupName : names) {
        auto id = groupName.first;
        const auto& name = groupName.second;
        const TopologyType type = (id == TopologyGroupId::HydroLn || id == TopologyGroupId::TransportRailway)
            ? TopologyType::Linear
            : TopologyType::Contour;
        groups_.insert({id,
            std::make_shared<TopologyGroup>(id, name, ftTypes.at(id), type)});
    }
    for (const auto& group : groups_) {
        groupsByNames_.insert({group.second->name(), group.second});
    }
}

TopologyGroupPtr TopologyGroupsRegistry::operator [] (TopologyGroupId groupId) const
{
    try {
        return groups_.at(groupId);
    } catch (const std::out_of_range& e) {
        throw UnknownTopologyGroupException() << "Invalid group id";
    }
}

TopologyGroupPtr TopologyGroupsRegistry::operator [] (const std::string& groupName) const
{
    try {
        return groupsByNames_.at(groupName);
    } catch (const std::out_of_range& e) {
        throw UnknownTopologyGroupException()
            << "Group with name " << groupName << " is not defined";
    }
}

TopologyGroups TopologyGroupsRegistry::allGroups() const
{
    TopologyGroups result;
    result.reserve(groups_.size());
    for (const auto& group : groups_) {
        result.push_back(group.second);
    }
    return result;
}


const TopologyGroupsRegistry& topologyGroupsRegistry()
{
    static const TopologyGroupsRegistry registry;
    return registry;
}


} // namespace topology_fixer
} // namespace wiki
} // namespace maps
