#include "geom_helpers.h"

#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/projection.h>

#include <geos/geom/Coordinate.h>
#include <geos/geom/LineString.h>
#include <geos/geom/LinearRing.h>
#include <geos/geom/GeometryFactory.h>
#include <geos/geom/CoordinateSequenceFactory.h>
#include <geos/geom/CoordinateArraySequenceFactory.h>
#include <geos/operation/IsSimpleOp.h>

#include <sstream>

namespace gl = maps::geolib3;
namespace gs = geos::geom;

namespace maps {
namespace wiki {
namespace topology_fixer {

template <>
gl::Polyline2 geomFromWkb<gl::Polyline2>(const std::string& wkb, SRID srid)
{
    std::istringstream is(wkb);
    gl::PointsVector points = gl::WKB::read<gl::Polyline2>(is).points();
    if (srid == SRID::Mercator) {
        for (auto& p : points) {
            p = gl::geoPoint2Mercator(p);
        }
    }
    return gl::Polyline2(std::move(points));
}

template <>
gl::Point2 geomFromWkb<gl::Point2>(const std::string& wkb, SRID srid)
{
    std::istringstream is(wkb);
    auto point = gl::WKB::read<gl::Point2>(is);
    return srid == SRID::Mercator ? gl::geoPoint2Mercator(point) : point;
}

template <>
std::string geomToWkb<gl::Point2>(const gl::Point2& point, SRID srid)
{
    std::ostringstream os;
    gl::WKB::write(
        srid == SRID::Mercator ? gl::mercator2GeoPoint(point) : point,
        os);
    return os.str();
}

template <>
std::string geomToWkb<gl::Polyline2>(const gl::Polyline2& linestring, SRID srid)
{
    std::ostringstream os;
    gl::PointsVector points = linestring.points();
    if (srid == SRID::Mercator) {
        for (auto& p : points) {
            p = gl::mercator2GeoPoint(p);
        }
    }
    gl::WKB::write(gl::Polyline2(std::move(points)), os);
    return os.str();
}


double
angle(const gl::Line2& line, const gl::Line2& other)
{
    const double dirAngle = gl::angle(line.direction(), other.direction());
    return dirAngle < M_PI / 2 ? dirAngle : M_PI - dirAngle;
}

bool
segmentsOverlap(const gl::Segment2& segment, const gl::Segment2& other,
    double eps, double maxAngle)
{
    if (gl::distance(segment, other) > eps ||
        angle(segment.line(), other.line()) > maxAngle)
    {
        return false;
    }
    double startProjFactor = gl::projectionFactor(segment, other.start());
    gl::Point2 startProj = gl::projection(segment.line(), other.start());

    double endProjFactor = gl::projectionFactor(segment, other.end());
    gl::Point2 endProj = gl::projection(segment.line(), other.end());

    if (startProjFactor > endProjFactor) {
        std::swap(startProjFactor, endProjFactor);
        std::swap(startProj, endProj);
    }

    const double epsFactor = eps / gl::length(segment);
    if (startProjFactor > 1 - epsFactor || endProjFactor < epsFactor) {
        return false;
    }

     return true;
}


bool
isSimple(const gl::PointsVector& points, bool requiredToBeClosed, bool silent)
{
    std::unique_ptr<std::vector<gs::Coordinate>> coords(
        new std::vector<gs::Coordinate>());
    coords->reserve(points.size());
    for (const auto& point : points) {
        coords->emplace_back(point.x(), point.y());
    }
    try {
        using gs::GeometryFactory;
        std::unique_ptr<gs::CoordinateSequence> cs(
            gs::DefaultCoordinateSequenceFactory().create(coords.release()));
        std::unique_ptr<gs::Geometry> geom(requiredToBeClosed
            ? GeometryFactory::getDefaultInstance()->createLinearRing(cs.release())
            : GeometryFactory::getDefaultInstance()->createLineString(cs.release()));
        geos::operation::IsSimpleOp op(*geom);
        return op.isSimple();
    } catch (const std::exception& e) {
        if (!silent) {
            WARN() << "Geom not simple: " << e.what();
        }
    } catch (...) {
        if (!silent) {
            WARN() << "Geom not simple: unknown error";
        }
    }

    return false;
}


double
signedArea(const gl::PointsVector& points)
{
    size_t npoints = points.size();
    if (npoints < 3) {
        return 0.0;
    }
    double sum = 0.0;
    for (size_t i = 0; i < npoints; ++i) {
        size_t j = (i + 1) % npoints;
        sum += (points[j].x() + points[i].x()) *
            (points[j].y() - points[i].y());
    }
    return sum / 2.0;
}

double
perimeter(const gl::PointsVector& points)
{
    return gl::length(gl::Polyline2(points));
}

double
perimeter(const gl::Polyline2& polyline)
{
    return gl::length(polyline);
}

geolib3::PointsVector
buildPolygon(FaceId faceId, const PlainEdgesData& edgesData)
{
    if (edgesData.empty()) {
        throw InvalidFaceException()
            << "Face " << faceId << " is empty";
    }

    geolib3::PointsVector result;

    std::unordered_map<NodeId, IdSet> nodeToEdges;

    for (const auto& edgeData : edgesData) {
        nodeToEdges[edgeData.second.fnodeId].insert(edgeData.first);
        nodeToEdges[edgeData.second.tnodeId].insert(edgeData.first);
    }

    if (nodeToEdges.size() != edgesData.size()) {
        std::ostringstream os;
        for (const auto& nodeEdges : nodeToEdges) {
            os << "Node: " << nodeEdges.first << ", edges: [";
            for (auto edgeId : nodeEdges.second) {
                os << edgeId << " ";
            }
            os << "]";
        }
        throw InvalidFaceException()
            << "Face " << faceId << " has " << edgesData.size() << " edges "
            << " but " << nodeToEdges.size() << " nodes, must be equal, got " << os.str();
    }
    for (const auto& nodeToEdge : nodeToEdges) {
        if (nodeToEdge.second.size() != 2) {
            std::ostringstream os;
            for (const auto& nodeEdges : nodeToEdges) {
                os << "Node: " << nodeEdges.first << ", edges: [";
                for (auto edgeId : nodeEdges.second) {
                    os << edgeId << " ";
                }
                os << "]";
            }
            throw InvalidFaceException() << "Face " << faceId
                << ", node " << nodeToEdge.first << " has less or more than "
                << "two incident edges corresponding to this face " << os.str();
        }
    }

    NodeId currentNodeId = nodeToEdges.begin()->first;
    EdgeId currentEdgeId = *nodeToEdges.begin()->second.begin();
    size_t visitedEdgesCount = 0;

    do {
        const auto& currentEdgeData = edgesData.at(currentEdgeId);
        NodeId nextNodeId = currentEdgeData.fnodeId == currentNodeId
            ? currentEdgeData.tnodeId
            : currentEdgeData.fnodeId;
        const auto& points = currentEdgeData.points;
        if (points.size() < 2) {
            throw InvalidFaceException() << "Edge " << currentEdgeId << " has too few points";
        }

        if (currentEdgeData.fnodeId == currentNodeId) { // forward
            ASSERT(result.empty() || result.back() == *points.begin());
            auto it = visitedEdgesCount > 0 ? std::next(points.begin()) : points.begin();
            std::copy(it, points.end(), std::back_inserter(result));
        } else {
            ASSERT(result.empty() || result.back() == *points.rbegin());
            auto it = visitedEdgesCount > 0 ? std::next(points.rbegin()) : points.rbegin();
            std::copy(it, points.rend(), std::back_inserter(result));
        }

        const auto& nextNodeEdges = nodeToEdges.at(nextNodeId);
        currentEdgeId = *nextNodeEdges.begin() == currentEdgeId
            ? *(++nextNodeEdges.begin())
            : *nextNodeEdges.begin();
        currentNodeId = nextNodeId;
    } while (++visitedEdgesCount != edgesData.size());

    if (result.size() < 4) {
        throw InvalidFaceException() << "Face " << faceId << " has too few points (must be >= 4)";
    }

    return result;
}

bool
isCycleOrCircuit(const TopologyData& data, const IdSet& edgeIds,
    NodeId fnodeId, NodeId tnodeId)
{
    if (edgeIds.empty()) {
        return false;
    }
    if (edgeIds.size() == 1) {
        const Edge& edge = data.edge(*edgeIds.begin());
        return edge.fnodeId() == fnodeId && edge.tnodeId() == tnodeId;
    }

    std::unordered_map<NodeId, IdSet> nodeToEdges;

    for (auto edgeId : edgeIds) {
        const Edge& edge = data.edge(edgeId);
        if (edge.fnodeId() == edge.tnodeId()) {
            return false;
        }
        nodeToEdges[edge.fnodeId()].insert(edgeId);
        nodeToEdges[edge.tnodeId()].insert(edgeId);
    }
    if ((fnodeId != tnodeId && nodeToEdges.size() != edgeIds.size() + 1) ||
        (fnodeId == tnodeId && nodeToEdges.size() != edgeIds.size()))
    {
        return false;
    }
    NodeId currentNodeId = fnodeId;
    IdSet visitedNodes;
    for (size_t i = 0; i < edgeIds.size(); ++i) {
        if (visitedNodes.count(currentNodeId) &&
            (fnodeId != tnodeId || i < edgeIds.size() - 1))
        {
            return false;
        }
        const auto& nodeEdgesIt = nodeToEdges.find(currentNodeId);
        if (nodeEdgesIt == nodeToEdges.end() ||
            nodeEdgesIt->second.empty() || nodeEdgesIt->second.size() > 2)
        {
            return false;
        }
        const EdgeId edgeId = *nodeEdgesIt->second.begin();
        const Edge& edge = data.edge(edgeId);
        const NodeId nextNodeId = edge.fnodeId() == currentNodeId
            ? edge.tnodeId()
            : edge.fnodeId();
        visitedNodes.insert(currentNodeId);
        auto curNodeIt = nodeToEdges.find(currentNodeId);
        ASSERT(curNodeIt != nodeToEdges.end());
        curNodeIt->second.erase(edgeId);
        if (curNodeIt->second.empty()) {
            nodeToEdges.erase(curNodeIt);
        }
        auto nextNodeIt = nodeToEdges.find(nextNodeId);
        ASSERT(nextNodeIt != nodeToEdges.end());
        nextNodeIt->second.erase(edgeId);
        if (nextNodeIt->second.empty()) {
            nodeToEdges.erase(nextNodeIt);
        }
        currentNodeId = nextNodeId;
    }
    return nodeToEdges.empty() && currentNodeId == tnodeId;
}

namespace {

bool
isFaceGeometryAllowed(
    const TopologyData& data,
    FaceId faceId,
    const PlainEdgesData& movedEdgesData,
    const IdSet& removedEdgeIds)
{
    auto faceEdges = data.face(faceId).edgeIds();
    if (faceEdges.size() == 1) {
        auto it = movedEdgesData.find(*faceEdges.begin());
        if (it != movedEdgesData.end() && !isSimple(it->second.points, true)) {
            return false;
        }
    } else {
        PlainEdgesData edgesData;
        for (auto eId : faceEdges) {
            if (removedEdgeIds.count(eId)) {
                continue;
            }
            auto it = movedEdgesData.find(eId);
            if (it != movedEdgesData.end()) {
                edgesData.insert(*it);
            } else {
                const auto& edge = data.edge(eId);
                const auto fnodeId = edge.fnodeId();
                const auto tnodeId = edge.tnodeId();
                edgesData.emplace(eId, PlainEdgeData{fnodeId, tnodeId, edge.linestring().points()});
            }
        }
        try {
            geolib3::Polygon2(buildPolygon(faceId, edgesData));
        } catch (const std::exception& e) {
            WARN() << "Face " << faceId << " new geometry is invalid " << e.what();
            return false;
        }
    }
    return true;
}

struct CheckEdgeData {
    NodeId fnodeId;
    NodeId tnodeId;
    geolib3::PointsVector points;
};

typedef std::unordered_map<EdgeId, CheckEdgeData> EdgesDataMap;

struct EdgesDataAfterNodeMove {
    EdgesDataMap movedEdgesData;
    IdSet removedEdgeIds;
};

boost::optional<EdgesDataAfterNodeMove>
edgesDataAfterNodeMoving(
    const TopologyData& data, NodeId toLeaveId, NodeId toRemoveId,
    double maxRemovableEdgeLength)
{
    EdgesDataMap movedEdgesData;
    IdSet removedEdgeIds;
    const Node& removedNode = data.node(toRemoveId);
    const Node& leftNode = data.node(toLeaveId);
    for (auto edgeId : removedNode.edgeIds()) {
        const Edge& edge = data.edge(edgeId);
        const auto& edgeGeom = edge.linestring();
        REQUIRE(edgeGeom.pointsNumber() >= 2, "Too few points in edge " << edgeId);
        CheckEdgeData d = {edge.fnodeId(), edge.tnodeId(), edgeGeom.points()};
        if (d.fnodeId == toRemoveId) {
            d.fnodeId = toLeaveId;
            d.points.front() = leftNode.point();
        }
        if (d.tnodeId == toRemoveId) {
            d.tnodeId = toLeaveId;
            d.points.back() = leftNode.point();
        }
        if (edge.fnodeId() != edge.tnodeId() && d.fnodeId == d.tnodeId) {
            if (gl::length(edgeGeom) < maxRemovableEdgeLength) {
                removedEdgeIds.insert(edgeId);
            } else if (data.ftGroup()->topologyType() == TopologyType::Contour) {
                return boost::none;
            } else {
                movedEdgesData.insert({edgeId, d});
            }
        } else {
            movedEdgesData.insert({edgeId, d});
        }
    }
    return EdgesDataAfterNodeMove{std::move(movedEdgesData), std::move(removedEdgeIds)};
}

bool
isContourNodesMergingAllowed(
    const TopologyData& data, NodeId toLeaveId, NodeId toRemoveId,
    double maxRemovableEdgeLength)
{
    IdSet faceIds;
    for (auto edgeId : data.node(toRemoveId).edgeIds()) {
        const Edge& edge = data.edge(edgeId);
        ASSERT(edge.isContour());
        const auto& edgeFaces = edge.faceIds();
        faceIds.insert(edgeFaces.begin(), edgeFaces.end());
    }
    boost::optional<EdgesDataAfterNodeMove> movedEdgesData = edgesDataAfterNodeMoving(
        data, toLeaveId, toRemoveId, maxRemovableEdgeLength);
    if (!movedEdgesData) {
        return false;
    }
    PlainEdgesData ed;
    for (const auto& d : movedEdgesData->movedEdgesData) {
        const auto edgeId = d.first;
        const auto& edgeData = d.second;
        ed.insert({edgeId, {edgeData.fnodeId, edgeData.tnodeId, edgeData.points}});
    }
    for (auto faceId : faceIds) {
        if (!isFaceGeometryAllowed(data, faceId, ed, movedEdgesData->removedEdgeIds)) {
            return false;
        }
    }
    return true;
}

bool
isLinearNodesMergingAllowed(
    const TopologyData& data, NodeId toLeaveId, NodeId toRemoveId,
    double maxRemovableEdgeLength)
{
    const auto& edgeIds = data.node(toRemoveId).edgeIds();
    for (auto edgeId : edgeIds) {
        const Edge& edge = data.edge(edgeId);
        ASSERT(edge.isLinear());
    }
    boost::optional<EdgesDataAfterNodeMove> movedEdgesData = edgesDataAfterNodeMoving(
        data, toLeaveId, toRemoveId, maxRemovableEdgeLength);
    if (!movedEdgesData) {
        return false;
    }
    for (const auto& d : movedEdgesData->movedEdgesData) {
        const auto edgeId = d.first;
        const auto& edgeData = d.second;
        if (!isEdgeGeometryAllowed(data, edgeId, edgeData.points)) {
            return false;
        }
    }
    return true;
}

} // namespace


bool
isNodesMergingAllowed(
    const TopologyData& data, NodeId toLeaveId, NodeId toRemoveId,
    double maxRemovableEdgeLength)
{
    if (data.isNodeLinear(toLeaveId) && data.isNodeLinear(toRemoveId)) {
        return isLinearNodesMergingAllowed(data, toLeaveId, toRemoveId,
            maxRemovableEdgeLength);
    }
    if (data.isNodeContour(toLeaveId) && data.isNodeContour(toRemoveId)) {
        return isContourNodesMergingAllowed(data, toLeaveId, toRemoveId,
            maxRemovableEdgeLength);
    }
    return false;
}

bool
isEdgeGeometryAllowed(
    const TopologyData& data,
    EdgeId edgeId,
    const geolib3::PointsVector& newGeom)
{
    const Edge& edge = data.edge(edgeId);
    PlainEdgesData ed = {{edgeId, {edge.fnodeId(), edge.tnodeId(), newGeom}}};
    if (data.ftGroup()->topologyType() == TopologyType::Contour) {
        for (auto faceId : edge.faceIds()) {
            if (!isFaceGeometryAllowed(data, faceId, ed, {})) {
                return false;
            }
        }
    } else {
        ASSERT(data.ftGroup()->topologyType() == TopologyType::Linear);
        if (!isSimple(newGeom, edge.fnodeId() == edge.tnodeId())) {
            return false;
        }
    }
    return true;
}

bool
areFacesSame(
    FaceId /*faceId1*/, const geolib3::PointsVector& points1,
    FaceId /*faceId2*/, const geolib3::PointsVector& points2,
    double tolerance,
    double maxAngleBetweenSegments,
    double maxTranslationFactor,
    double maxAPRatioFactor)
{
    if (points1.empty() || points2.empty()) {
        return false;
    }
    gl::Polyline2 extRing1(points1);
    gl::Polyline2 extRing2(points2);

    gl::BoundingBox bbox1 = extRing1.boundingBox();
    gl::BoundingBox bbox2 = extRing2.boundingBox();
    if (gl::sign(bbox1.width() - bbox2.width(), 2 * tolerance)
        || gl::sign(bbox1.height() - bbox2.height(), 2 * tolerance))
    {
        return false;
    }
    gl::Vector2 translationDelta = bbox2.center() - bbox1.center();
    if (gl::length(translationDelta) / bbox1.diagonalLength() > maxTranslationFactor) {
        return false;
    }
    const double perimeter1 = perimeter(extRing1);
    const double perimeter2 = perimeter(extRing2);
    if (perimeter1 < tolerance || perimeter2 < tolerance) {
        return false;
    }
    const double apRatio1 = std::fabs(topology_fixer::signedArea(extRing1.points())) / perimeter1;
    const double apRatio2 = std::fabs(topology_fixer::signedArea(extRing2.points())) / perimeter2;
    if (2 * std::fabs(apRatio1 - apRatio2) / (apRatio1 + apRatio2) > maxAPRatioFactor) {
        return false;
    }
    bool allMatchesFound = true;
    for (size_t i = 0; i < allMatchesFound && extRing1.segmentsNumber(); ++i) {
        auto seg1 = extRing1.segmentAt(i);
        if (gl::length(seg1) < tolerance) {
            continue;
        }
        bool matchFound = false;
        for (size_t j = 0; !matchFound && j < extRing2.segmentsNumber(); ++j) {
            auto seg2 = extRing2.segmentAt(j);
            if (gl::length(seg2) < tolerance ||
                angle(seg1.line(), seg2.line()) > maxAngleBetweenSegments)
            {
                continue;
            }
            gl::Segment2 trSeg1 = seg1 + translationDelta;
            matchFound =
                gl::distance(seg2.line(), trSeg1.start()) < tolerance &&
                gl::distance(seg2.line(), trSeg1.end()) < tolerance &&
                gl::distance(trSeg1.line(), seg2.start()) < tolerance &&
                gl::distance(trSeg1.line(), seg2.end()) < tolerance;
        }
        allMatchesFound = allMatchesFound && matchFound;
    }
    return allMatchesFound;
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
