#include "lineal_edges_intersector.h"
#include "segments/segment_endpoints_snapper.h"
#include "segments/segments_intersector.h"
#include "segments/segments_overlapper.h"
#include "topology_data_helpers.h"
#include "geom_helpers.h"

#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/projection.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/intersection.h>

#include <initializer_list>
#include <algorithm>
#include <deque>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

std::unique_ptr<EdgesFixer::ProcessingData>
LinearEdgesIntersectionsFixer::initProcessingData(EdgeId edgeId1, EdgeId edgeId2)
{
    return std::unique_ptr<EdgesFixer::ProcessingData>(
        new ProcessingData(edgeId1, edgeId2, data_, locker_, params_));
}

SegmentsProcessingResult
LinearEdgesIntersectionsFixer::ProcessingData::processEdges()
{
    ASSERT(edgesData.size() == 2);

    auto result = processOverlaps(
        segmentsGraph, edgesData.at(edgeId1), edgesData.at(edgeId2),
        params_.overlapTolerance, params_.maxAngleBetweenOverlappingSegments);
    bool hasChanges = result.hasChanges;

    result = processSnappingPoints(
        segmentsGraph, result.segmentIds1, result.segmentIds2,
        params_.maxSnapDistance, params_.snapTolerance);
    hasChanges = hasChanges || result.hasChanges;

    auto invertedSnapResult = processSnappingPoints(
        segmentsGraph, result.segmentIds2, result.segmentIds1,
        params_.maxSnapDistance, params_.snapTolerance);
    hasChanges = hasChanges || invertedSnapResult.hasChanges;
    result.segmentIds1 = invertedSnapResult.segmentIds2;
    result.segmentIds2 = invertedSnapResult.segmentIds1;

    result = processIntersections(
        segmentsGraph, result.segmentIds1, result.segmentIds2);
    result.hasChanges = hasChanges || result.hasChanges;

    return result;
}

bool
LinearEdgesIntersectionsFixer::ProcessingData::newNodesValid() const
{
    for (const auto& pointDegree : segmentsGraph.pointDegrees()) {
        const auto pointId = pointDegree.first;
        const auto incidencesType = segmentsGraph.pointIncidencesType(pointId);
        if (incidencesType == SegmentsGraph::PointIncidencesType::Isolated) {
            return false;
        }
    }
    return true;
}

void
LinearEdgesIntersectionsFixer::ProcessingData::createNewNodes()
{
    for (const auto& pointDegree : segmentsGraph.pointDegrees()) {
        const auto pointId = pointDegree.first;
        auto& point = segmentsGraph.point(pointId);
        const auto incidencesType = segmentsGraph.pointIncidencesType(pointId);
        if (!point.nodeId && (pointDegree.second > 2 ||
            incidencesType == SegmentsGraph::PointIncidencesType::Mixed))
        {
            const auto newNodeId = dataCopy->addNode(point.pos);
            nodeIdToPointIdMap.insert({newNodeId, pointId});
            point.nodeId = newNodeId;
        }
    }
}

bool
LinearEdgesIntersectionsFixer::ProcessingData::restoreTopologyInCopy()
{
    IdSet validatedEdgeIds;
    for (const auto& edgeData : edgesData) {
        EdgeIdsSet newEdgeIds;
        for (auto segId : edgeData.second) {
            const auto& seg = segmentsGraph.segment(segId);
            ASSERT(seg.edgeId);
            newEdgeIds.insert(*seg.edgeId);
        }
        ASSERT(!newEdgeIds.empty());
        originalToSplittedEdgesMap.insert({edgeData.first, newEdgeIds});
        if (newEdgeIds.size() > 1 || *newEdgeIds.begin() != edgeData.first) {
            IdSet edgeIds(newEdgeIds.begin(), newEdgeIds.end());
            dataCopy->replaceEdge(edgeData.first, edgeIds);
            validatedEdgeIds.insert(newEdgeIds.begin(), newEdgeIds.end());
        }
    }
    for (const auto& edgeData : edgesData) {
        const EdgeId edgeId = edgeData.first;
        const Edge& edge = originalDataCopy->edge(edgeId);
        const auto& partIds = originalToSplittedEdgesMap.at(edgeId);
        IdSet unorderedPartIds(partIds.begin(), partIds.end());
        if (!isCycleOrCircuit(*dataCopy, unorderedPartIds, edge.fnodeId(), edge.tnodeId())) {
            WARN() << "Edge " << edgeId << " does not form circuit or cycle after splitting";
            return false;
        }
    }
    for (auto edgeId : validatedEdgeIds) {
        const Edge& edge = dataCopy->edge(edgeId);
        bool isClosed = edge.fnodeId() == edge.tnodeId();
        if (!isSimple(edge.linestring().points(), isClosed)) {
            WARN() << "Edge " << edgeId << " geometry is not simple";
            return false;
        }
    }
    return true;
}

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
