#pragma once

#include "../topology_data.h"

#include <maps/libs/geolib/include/polygon.h>

#include <map>
#include <mutex>

namespace maps {
namespace wiki {
namespace topology_fixer {

struct Area {
    Area() : exterior(0.0), interior(0.0) {}

    double exterior;
    double interior;
};

inline
boost::optional<double>
faceArea(const TopologyData& data, FaceId faceId)
{
    try {
        geolib3::Polygon2 poly(data.faceVertices(faceId));
        return std::fabs(poly.area());
    } catch (const std::exception&)
    {}
    return boost::none;
}

inline
boost::optional<Area>
masterArea(const TopologyData& data, MasterId masterId)
{
    const Master& master = data.master(masterId);
    Area result;
    for (const auto& faceRel: master.faceRels()) {
        try {
            geolib3::Polygon2 poly(data.faceVertices(faceRel.first));
            (faceRel.second == FaceRelationType::Exterior
                ? result.exterior
                : result.interior) += std::fabs(poly.area());
        } catch (const std::exception&) {
            return boost::none;
        }
    }
    return result;
}

class MasterAreaHolder {
public:

    explicit MasterAreaHolder(const TopologyData& data)
    {
        std::map<FaceId, double> faceAreas;
        for (auto faceId : data.faceIds()) {
            auto faceAreaOpt = faceArea(data, faceId);
            if (faceAreaOpt) {
                faceAreas.insert({faceId, *faceAreaOpt});
            }
        }
        for (const auto& masterPair: data.masters()) {
            const Master& master = masterPair.second;
            Area area;
            bool allFacesValid = true;
            for (const auto& faceRel: master.faceRels()) {
                auto faceAreaIt = faceAreas.find(faceRel.first);
                if (faceAreaIt == faceAreas.end()) {
                    allFacesValid = false;
                    break;
                }
                (faceRel.second == FaceRelationType::Exterior
                    ? area.exterior
                    : area.interior) += faceAreaIt->second;
            }
            if (allFacesValid) {
                areas_.insert({master.id(), area});
            }
        }
    }

    boost::optional<Area> area(MasterId masterId)
    {
        std::lock_guard<std::mutex> guard(mutex_);
        auto it = areas_.find(masterId);
        if (it != areas_.end()) {
            return it->second;
        }
        return boost::none;
    }

    void setArea(MasterId masterId, const Area& area)
    {
        std::lock_guard<std::mutex> guard(mutex_);
        areas_[masterId] = area;
    }

private:


    std::map<MasterId, Area> areas_;
    std::mutex mutex_;
};

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
