#include "segment_endpoints_snapper.h"

#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/projection.h>

#include <maps/libs/log8/include/log8.h>

#include <limits>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

namespace {

boost::optional<PointId>
closestPointId(const SegmentsGraph& graph,
    const gl::Segment2& segment, const PointIdsList& pointIds,
    double maxSnapDistance)
{
    boost::optional<PointId> closest;
    auto minDist = std::numeric_limits<double>::infinity();
    for (auto pointId : pointIds) {
        const Point& point = graph.point(pointId);
        const double dist = gl::distance(segment, point.pos);
        if (dist > maxSnapDistance) {
            continue;
        }

        if (!closest || dist < minDist) {
            closest = pointId;
            minDist = dist;
        }
    }
    return closest;
}

} // namespace

SegmentsProcessingResult
SegmentsPairEndpointsSnapper::operator () (
    SegmentsGraph& graph, SegmentId segmentId1, SegmentId segmentId2) const
{
    gl::Segment2 segmentGeom1 = graph.segmentGeom(segmentId1);
    gl::Segment2 segmentGeom2 = graph.segmentGeom(segmentId2);
    const double segmentLength = gl::length(segmentGeom1);
    const double otherSegmentLength = gl::length(segmentGeom2);
    if (segmentId1 == segmentId2 ||
        segmentLength < tolerance_ || otherSegmentLength < tolerance_ ||
        graph.haveCommonEndpoint(segmentId1, segmentId2))
    {
        return {{segmentId1}, {segmentId2}, false};
    }
    const auto& segment1 = graph.segment(segmentId1);
    const auto& segment2 = graph.segment(segmentId2);
    auto closestId = closestPointId(graph, segmentGeom1,
        {segment2.startPointId, segment2.endPointId}, maxSnapDistance_);
    if (!closestId) {
        return {{segmentId1}, {segmentId2}, false};
    }
    const Point& point = graph.point(*closestId);
    const auto projFactor = gl::projectionFactor(segmentGeom1, point.pos);
    const double epsFactor = tolerance_ / segmentLength;
    if (projFactor < -epsFactor || projFactor > 1 + epsFactor) {
        return {{segmentId1}, {segmentId2}, false};
    }
    boost::optional<PointId> mergePointId;
    if (std::fabs(projFactor) < epsFactor) {
        if (!graph.pointsShareSegment(point.id, segment1.startPointId)) {
            mergePointId = segment1.startPointId;
        } else {
            return {{segmentId1}, {segmentId2}, false};
        }
    } else if (std::fabs(projFactor - 1) < epsFactor) {
        if (!graph.pointsShareSegment(point.id, segment1.endPointId)) {
            mergePointId = segment1.endPointId;
        } else {
            return {{segmentId1}, {segmentId2}, false};
        }
    }
    if (mergePointId) {
        if (!point.nodeId) {
            graph.mergePoints(point.id, *mergePointId);
            return {{segmentId1}, {segmentId2}, true};
        }
    if (!graph.point(*mergePointId).nodeId) {
            graph.mergePoints(*mergePointId, point.id);
            return {{segmentId1}, {segmentId2}, true};
        }
        return {{segmentId1}, {segmentId2}, false};
    }
    const auto startId = segment1.startPointId;
    const auto endId = segment1.endPointId;
    graph.removeSegment(segmentId1);
    return {
        {graph.addSegment(startId, *closestId).id, graph.addSegment(*closestId, endId).id},
        {segmentId2},
        true};
}

void
SegmentEndpointsSnapper::Impl::doAfterProcessing()
{
    removeDuplicatedSegments();
}

SegmentsProcessingResult
processSnappingPoints(SegmentsGraph& graph,
    const SegmentIdsList& segmentIds, const SegmentIdsList& otherSegmentIds,
    double maxSnapDistance, double tolerance)
{
    SegmentEndpointsSnapper snapper(maxSnapDistance, tolerance);
    return snapper(graph, segmentIds, otherSegmentIds);
}

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
