#pragma once

#include "../../common.h"

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/segment.h>

#include <boost/optional.hpp>

#include <list>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

namespace gl = maps::geolib3;

typedef DBIdType PointId;
typedef DBIdType SegmentId;
typedef IdSet SegmentIdsSet;
typedef std::list<SegmentId> SegmentIdsList;
typedef std::list<PointId> PointIdsList;

struct Point {
    PointId id;
    boost::optional<NodeId> nodeId;
    gl::Point2 pos;
    SegmentIdsSet segmentIds;
};

struct Segment {
    SegmentId id;
    PointId startPointId;
    PointId endPointId;
    boost::optional<EdgeId> edgeId;
    bool isShared;
};

class SegmentsGraph {
public:
    typedef std::unordered_map<PointId, size_t> DegreesMap;

    SegmentsGraph() : gen_(std::make_shared<IdGenerator>(0)) {}

    const Point& addPoint(boost::optional<NodeId> nodeId, const gl::Point2& pos);
    const Segment& addSegment(PointId startPointId, PointId endPointId);

    const Point& point(PointId id) const { return points_.at(id); }
    Point& point(PointId id) { return points_.at(id); }

    const Segment& segment(SegmentId id) const { return segments_.at(id); }
    Segment& segment(SegmentId id) { return segments_.at(id); }

    gl::Segment2 segmentGeom(SegmentId segmentId) const;

    DegreesMap pointDegrees() const;

    // Asserts point is isolated
    void removeIsolatedPoint(PointId pointId);

    // Does not remove points which became isolated
    void removeSegment(SegmentId segmentId);
    // Does not remove points which became isolated
    void moveSegment(SegmentId segmentId, PointId fromId, PointId toId);

    bool haveCommonEndpoint(SegmentId segmentId1, SegmentId segmentId2) const;

    typedef std::unordered_map<SegmentId, SegmentId> OrigToUnitedSegmentIdsMap;

    OrigToUnitedSegmentIdsMap uniteDuplicatedSegments();

    bool pointsShareSegment(PointId id1, PointId id2) const
    {
        const Point& point1 = points_.at(id1);
        const Point& point2 = points_.at(id2);
        for (auto segId : point1.segmentIds) {
            if (point2.segmentIds.count(segId)) {
                return true;
            }
        }
        return false;
    }

    SegmentIdsSet sharedSegmentIds(PointId id1, PointId id2) const
    {
        SegmentIdsSet ids;
        const Point& point1 = points_.at(id1);
        const Point& point2 = points_.at(id2);
        for (auto segId : point1.segmentIds) {
            if (point2.segmentIds.count(segId)) {
                ids.insert(segId);
            }
        }
        return ids;
    }

    /// @param fromId point is removed
    void mergePoints(PointId fromId, PointId toId);

    enum class PointIncidencesType { Isolated, SharedOnly, NonSharedOnly, Mixed };

    PointIncidencesType pointIncidencesType(PointId pointId) const;

    PointId otherPointId(const Segment& segment, PointId pointId) const;

    SegmentIdsList sharedSegmentIds() const;

    struct Circuit {
        SegmentIdsList segmentIdsList;
        PointIdsList pointIdsList;
    };

    std::list<Circuit> buildPartition() const;

    void print() const;

private:
    typedef std::unordered_map<PointId, SegmentIdsSet> PointIncidencesMap;

    PointIncidencesMap
    mergeablePointIncidences() const;

    typedef std::unordered_map<PointId, Point> Points;
    typedef std::unordered_map<SegmentId, Segment> Segments;

    std::shared_ptr<IdGenerator> gen_;
    Points points_;
    Segments segments_;
};

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
