#include "segments_processing_common.h"
#include "../unique_list.h"

#include <maps/libs/log8/include/log8.h>

#include <deque>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

namespace {

std::set<PointId>
segmentPoints(const SegmentsGraph& graph, const SegmentIdsList& segmentIds)
{
    std::set<PointId> pointIds;
    for (auto segmentId : segmentIds) {
        const Segment& segment = graph.segment(segmentId);
        pointIds.insert(segment.startPointId);
        pointIds.insert(segment.endPointId);
    }
    return pointIds;
}

} // namespace

SegmentsProcessingResult
SegmentsProcessor::operator () (
    SegmentsGraph& graph,
    const SegmentIdsList& segmentIds,
    const SegmentIdsList& otherSegmentIds) const
{
    std::set<PointId> pointIdsBefore1 = segmentPoints(graph, segmentIds);
    std::set<PointId> pointIdsBefore2 = segmentPoints(graph, otherSegmentIds);
    std::unique_ptr<Impl> impl = initImpl(graph, segmentIds, otherSegmentIds);
    SegmentsProcessingResult result = impl->doProcessing();
    std::set<PointId> pointIdsAfter1 = segmentPoints(graph, result.segmentIds1);
    std::set<PointId> pointIdsAfter2 = segmentPoints(graph, result.segmentIds2);
    result.hasChanges = result.hasChanges ||
        !idsSymDiff(pointIdsBefore1, pointIdsAfter1).empty() ||
        !idsSymDiff(pointIdsBefore2, pointIdsAfter2).empty() ||
        !idsSymDiff(segmentIds, result.segmentIds1).empty() ||
        !idsSymDiff(otherSegmentIds, result.segmentIds2).empty();
    return result;
}

namespace {

typedef UniqueList<SegmentId> SegmentsUniqueList;

} // namespace

SegmentsProcessingResult
SegmentsProcessor::Impl::processAllInteractions()
{
    bool hasChanges = false;

    doBeforeProcessing();
    SegmentIdsList thisSegmentsList = segmentIds_;
    SegmentIdsList otherSegmentsList= otherSegmentIds_;

    auto commonIdsList = idsIntersection(thisSegmentsList, otherSegmentsList);
    IdSet commonIds(commonIdsList.begin(), commonIdsList.end());

    auto thisSegments = SegmentsUniqueList(std::move(thisSegmentsList));
    auto otherSegments = SegmentsUniqueList(std::move(otherSegmentsList));

    for (auto it = thisSegments.begin(); it != thisSegments.end(); ) {
        const auto segmentId = *it;
        bool modified = false;
        for (auto otherId : otherSegments) {
            if (commonIds.count(segmentId) || commonIds.count(otherId)) {
                continue;
            }
            auto splitResult = processor_(graph_, segmentId, otherId);
            hasChanges = hasChanges || splitResult.hasChanges;
            if (!splitResult.hasChanges) {
                continue;
            }
            const auto& thisSegmentChunks = splitResult.segmentIds1;
            const auto& otherSegmentChunks = splitResult.segmentIds2;
            ASSERT(!thisSegmentChunks.empty() && !otherSegmentChunks.empty());
            it = thisSegments.replace(segmentId, thisSegmentChunks);
            std::advance(it, -thisSegmentChunks.size());
            otherSegments.replace(otherId, otherSegmentChunks);
            auto newCommonIds = idsIntersection(thisSegmentChunks, otherSegmentChunks);
            commonIds.insert(newCommonIds.begin(), newCommonIds.end());
            modified = true;
            break;
        }
        if (!modified) {
            ++it;
        }
    }

    segmentIds_ = SegmentIdsList(thisSegments.begin(), thisSegments.end());
    otherSegmentIds_ = SegmentIdsList(otherSegments.begin(), otherSegments.end());

    doAfterProcessing();

    return {segmentIds_, otherSegmentIds_, hasChanges};
}

struct SegmentQueueElement {
    // iterator to edge in first list
    SegmentsUniqueList::Iterator thisIt;
    // segment id in second list to start check
    SegmentId otherStartId;
};

typedef std::deque<SegmentQueueElement> SegmentsQueue;

SegmentsQueue
buildQueue(SegmentsUniqueList& segmentIds, SegmentsUniqueList& otherSegmentIds)
{
    SegmentsQueue queue;
    REQUIRE(!otherSegmentIds.empty(), "Empty segments list");
    for (auto it = segmentIds.begin(); it != segmentIds.end(); ++it) {
        queue.push_back({it, *otherSegmentIds.begin()});
    }
    return queue;
}

SegmentsProcessingResult
SegmentsProcessor::Impl::processInteractionsOnce()
{
    bool hasChanges = false;

    doBeforeProcessing();

    SegmentIdsList thisSegmentsList = segmentIds_;
    SegmentIdsList otherSegmentsList= otherSegmentIds_;

    auto commonIdsList = idsIntersection(thisSegmentsList, otherSegmentsList);
    IdSet commonIds(commonIdsList.begin(), commonIdsList.end());

    auto thisSegments = SegmentsUniqueList(std::move(thisSegmentsList));
    auto otherSegments = SegmentsUniqueList(std::move(otherSegmentsList));

    SegmentsQueue queue = buildQueue(thisSegments, otherSegments);

    while (!queue.empty()) {
        SegmentQueueElement currentElement = queue.front();
        queue.pop_front();
        const auto segmentId = *currentElement.thisIt;
        auto otherIt = otherSegments.find(currentElement.otherStartId);
        for ( ; otherIt != otherSegments.end(); ++otherIt) {
            const auto otherId = *otherIt;
            if (commonIds.count(segmentId) || commonIds.count(otherId)) {
                continue;
            }
            auto splitResult = processor_(graph_, segmentId, otherId);
            hasChanges = hasChanges || splitResult.hasChanges;
            if (!splitResult.hasChanges) {
                continue;
            }
            const auto& thisSegmentChunks = splitResult.segmentIds1;
            const auto& otherSegmentChunks = splitResult.segmentIds2;
            ASSERT(!thisSegmentChunks.empty() && !otherSegmentChunks.empty());
            auto nextIt = thisSegments.replace(segmentId, thisSegmentChunks);
            otherIt = otherSegments.replace(otherId, otherSegmentChunks);
            for (auto& queueEl : queue) {
                if (queueEl.otherStartId == otherId) {
                    queueEl.otherStartId = otherSegmentChunks.front();
                }
            }
            if (otherIt != otherSegments.end()) {
                auto it = nextIt;
                std::advance(it, -thisSegmentChunks.size());
                for (; it != nextIt; ++it) {
                    queue.push_back({it, *otherIt});
                }
            }
            auto newCommonIds = idsIntersection(thisSegmentChunks, otherSegmentChunks);
            commonIds.insert(newCommonIds.begin(), newCommonIds.end());
            break;
        }
    }

    segmentIds_ = SegmentIdsList(thisSegments.begin(), thisSegments.end());
    otherSegmentIds_ = SegmentIdsList(otherSegments.begin(), otherSegments.end());

    doAfterProcessing();

    return {segmentIds_, otherSegmentIds_, hasChanges};
}


namespace {

bool
segmentsAreDuplicate(const SegmentsGraph& graph, SegmentId id1, SegmentId id2)
{
    const Segment& seg1 = graph.segment(id1);
    const Segment& seg2 = graph.segment(id2);
    return (seg1.startPointId == seg2.startPointId &&
            seg1.endPointId == seg2.endPointId) ||
           (seg1.startPointId == seg2.endPointId &&
            seg1.endPointId == seg2.startPointId);
}

boost::optional<PointId>
segmentsDanglingCommonPointId(const SegmentsGraph& graph,
    SegmentId id1, SegmentId id2, const SegmentIdsSet& segmentIds)
{
    auto isDanglingPoint = [&] (PointId id) -> bool
    {
        auto incidentSegmentIds = graph.point(id).segmentIds;
        incidentSegmentIds.erase(id1);
        incidentSegmentIds.erase(id2);
        bool hasOtherIncidentSegments = false;
        for (auto segId : incidentSegmentIds) {
            if (segmentIds.count(segId)) {
                hasOtherIncidentSegments = true;
                break;
            }
        }
        return !hasOtherIncidentSegments;
    };
    const Segment& seg1 = graph.segment(id1);
    if (isDanglingPoint(seg1.startPointId)) {
        return seg1.startPointId;
    }
    if (isDanglingPoint(seg1.endPointId)) {
        return seg1.endPointId;
    }
    return boost::none;
}

void
removeSnagsImpl(SegmentsGraph& graph,
    SegmentIdsList& edgeSegments, const SegmentIdsSet& otherEdgeSegments)
{
    ASSERT(!edgeSegments.empty());
    SegmentIdsSet segmentIds(edgeSegments.begin(), edgeSegments.end());
    auto currentSegmentIt = edgeSegments.begin();
    auto nextSegmentIt = std::next(edgeSegments.begin());
    while (nextSegmentIt != edgeSegments.end()) {
        if (segmentsAreDuplicate(graph, *currentSegmentIt, *nextSegmentIt) &&
            !otherEdgeSegments.count(*currentSegmentIt) &&
            !otherEdgeSegments.count(*nextSegmentIt))
        {
            boost::optional<PointId> commonPointId =
                segmentsDanglingCommonPointId(
                    graph, *currentSegmentIt, *nextSegmentIt, segmentIds);
            if (!commonPointId) {
                currentSegmentIt = nextSegmentIt;
                ++nextSegmentIt;
                continue;
            }
            const Point& commonPoint = graph.point(*commonPointId);
            graph.removeSegment(*currentSegmentIt);
            graph.removeSegment(*nextSegmentIt);
            segmentIds.erase(*currentSegmentIt);
            segmentIds.erase(*nextSegmentIt);
            if (commonPoint.segmentIds.empty()) {
                graph.removeIsolatedPoint(*commonPointId);
            }
            nextSegmentIt = edgeSegments.erase(nextSegmentIt);
            currentSegmentIt = edgeSegments.erase(currentSegmentIt);
            if (edgeSegments.size() <= 2) {
                break;
            }
            if (currentSegmentIt == edgeSegments.begin()) {
                ++nextSegmentIt;
            } else {
                --currentSegmentIt;
            }
        } else {
            currentSegmentIt = nextSegmentIt;
            ++nextSegmentIt;
        }
    }
}

} // namespace


void
SegmentsProcessor::Impl::removeDuplicatedSegments()
{
    auto oldToUnitedSegmentIds = graph_.uniteDuplicatedSegments();
    auto replaceIds = [&] (SegmentIdsList& segmentIds)
    {
        for (auto& segmentId : segmentIds) {
            auto it = oldToUnitedSegmentIds.find(segmentId);
            if (it != oldToUnitedSegmentIds.end()) {
                segmentId = it->second;
            }
        }
    };
    replaceIds(segmentIds_);
    replaceIds(otherSegmentIds_);
}

void
SegmentsProcessor::Impl::removeSnags()
{
    SegmentIdsSet secondEdgeSegments(otherSegmentIds_.begin(), otherSegmentIds_.end());
    removeSnagsImpl(graph_, segmentIds_, secondEdgeSegments);
    SegmentIdsSet firstEdgeSegments(segmentIds_.begin(), segmentIds_.end());
    removeSnagsImpl(graph_, otherSegmentIds_, firstEdgeSegments);
}

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
