#pragma once

#include "../topology_data.h"
#include "topology_data_helpers.h"

#include <mutex>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

namespace gl = maps::geolib3;

class TopologyDataProxy {
public:
    explicit TopologyDataProxy(TopologyData& data)
        : data_(data)
    {}

    virtual ~TopologyDataProxy() = default;

    const TopologyGroupPtr& ftGroup() const { return data_.ftGroup(); }
    std::shared_ptr<IdGenerator> idGenerator() const { return data_.idGenerator(); }
    SRID srid() const { return data_.srid(); }

    virtual const Node& node(NodeId id) = 0;
    virtual const Edge& edge(EdgeId id) = 0;
    virtual const Face& face(FaceId id) = 0;
    virtual const Master& master(MasterId id) = 0;

    virtual bool nodeExists(NodeId id) = 0;

    virtual void addExistingNode(NodeId id, const gl::Point2& point) = 0;

    virtual void addExistingEdge(
        EdgeId id, gl::Polyline2 linestring,
        NodeId fnodeId, NodeId tnodeId,
        ZLevelType fZlev, ZLevelType tZlev) = 0;

    virtual void addExistingFace(FaceId faceId) = 0;
    virtual void addExistingMaster(MasterId masterId) = 0;
    virtual void addExistingMaster(MasterId masterId, const FtAttributes& ftAttrs) = 0;

    virtual void applyDiff(const TopologyData& fromData, const TopologyDataDiff& diff) = 0;

    virtual void replaceEdge(EdgeId edgeId, const IdSet& edgeIds) = 0;

    virtual void addFaceEdge(DBIdType faceId, DBIdType edgeId) = 0;

    virtual void addMasterFaceRel(
        DBIdType masterId, DBIdType faceId, FaceRelationType isInterior) = 0;
    virtual void addMasterEdgeRel(DBIdType masterId, DBIdType edgeId) = 0;

    virtual void removeMasterFaceRel(DBIdType masterId, DBIdType faceId) = 0;
    virtual void removeMasterEdgeRel(DBIdType masterId, DBIdType edgeId) = 0;

    virtual void removeFaceEdgeRel(DBIdType faceId, DBIdType edgeId) = 0;

    virtual void removeMaster(DBIdType id, DeletionMode mode) = 0;
    virtual void removeFace(DBIdType id, DeletionMode mode) = 0;
    virtual void removeEdge(DBIdType id, DeletionMode mode) = 0;
    virtual void removeNode(DBIdType id) = 0;

    virtual std::vector<geolib3::Point2> faceVertices(DBIdType faceId) = 0;

protected:
    TopologyData& data_;
};

/**
 * Trivial forwarding to corresponding TopologyData methods.
 */
class TrivialTopologyDataProxy : public TopologyDataProxy {
public:
    explicit TrivialTopologyDataProxy(TopologyData& data)
        : TopologyDataProxy(data)
    {}

    const Node& node(NodeId id) override { return data_.node(id); }
    const Edge& edge(EdgeId id) override { return data_.edge(id); }
    const Face& face(FaceId id) override { return data_.face(id); }
    const Master& master(MasterId id) override { return data_.master(id); }

    bool nodeExists(NodeId id) override { return data_.nodeExists(id); }

    void addExistingNode(NodeId id, const gl::Point2& point) override
    {
        data_.addExistingNode(id, point);
    }

    void addExistingEdge(
        EdgeId id, gl::Polyline2 linestring,
        NodeId fnodeId, NodeId tnodeId,
        ZLevelType fZlev, ZLevelType tZlev) override
    {
        data_.addExistingEdge(id, std::move(linestring), fnodeId, tnodeId, fZlev, tZlev);
    }

    void addExistingFace(FaceId faceId) override
    {
        data_.addExistingFace(faceId);
    }

    void addExistingMaster(MasterId masterId) override
    {
        data_.addExistingMaster(masterId);
    }

    void addExistingMaster(MasterId masterId, const FtAttributes& ftAttrs) override
    {
        data_.addExistingMaster(masterId, ftAttrs);
    }

    void applyDiff(const TopologyData& fromData, const TopologyDataDiff& diff) override
    {
        topology_fixer::applyDiff(data_, fromData, diff);
    }

    void replaceEdge(EdgeId edgeId, const IdSet& edgeIds) override
    {
        data_.replaceEdge(edgeId, edgeIds);
    }

    void addFaceEdge(DBIdType faceId, DBIdType edgeId) override
    {
        data_.addFaceEdge(faceId, edgeId);
    }

    void addMasterFaceRel(
        DBIdType masterId, DBIdType faceId, FaceRelationType isInterior) override
    {
        data_.addMasterFaceRel(masterId, faceId, isInterior);
    }

    void addMasterEdgeRel(
        DBIdType masterId, DBIdType edgeId) override
    {
        data_.addMasterEdgeRel(masterId, edgeId);
    }

    void removeMasterFaceRel(DBIdType masterId, DBIdType faceId) override
    {
        data_.removeMasterFaceRel(masterId, faceId);
    }

    void removeMasterEdgeRel(DBIdType masterId, DBIdType edgeId) override
    {
        data_.removeMasterEdgeRel(masterId, edgeId);
    }

    void removeFaceEdgeRel(DBIdType faceId, DBIdType edgeId) override
    {
        data_.removeFaceEdgeRel(faceId, edgeId);
    }

    void removeMaster(DBIdType id, DeletionMode mode) override
    {
        data_.removeMaster(id, mode);
    }

    void removeFace(DBIdType id, DeletionMode mode) override
    {
        data_.removeFace(id, mode);
    }

    void removeEdge(DBIdType id, DeletionMode mode) override
    {
        data_.removeEdge(id, mode);
    }

    void removeNode(DBIdType id) override
    {
        data_.removeNode(id);
    }

    std::vector<geolib3::Point2> faceVertices(DBIdType faceId) override
    {
        return data_.faceVertices(faceId);
    }
};

/**
 * Class only ensures TopologyData object itself is accesssed correctly.
 * It CAN NOT guarantee that different worker threads are not trying to
 *   modify same topological objects (nodes, edges etc.)
 *   so this is client's responsibility.
 */
class ThreadSafeTopologyDataProxy : public TopologyDataProxy {
public:
    explicit ThreadSafeTopologyDataProxy(TopologyData& data)
        : TopologyDataProxy(data)
    {}

    const Node& node(NodeId id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.node(id);
    }

    const Edge& edge(EdgeId id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.edge(id);
    }

    const Face& face(FaceId id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.face(id);
    }

    const Master& master(MasterId id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.master(id);
    }

    bool nodeExists(NodeId id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.nodeExists(id);
    }

    void addExistingNode(NodeId id, const gl::Point2& point) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addExistingNode(id, point);
    }

    void addExistingEdge(
        EdgeId id, gl::Polyline2 linestring,
        NodeId fnodeId, NodeId tnodeId,
        ZLevelType fZlev, ZLevelType tZlev) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addExistingEdge(id, std::move(linestring), fnodeId, tnodeId, fZlev, tZlev);
    }

    void addExistingFace(FaceId faceId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addExistingFace(faceId);
    }

    void addExistingMaster(MasterId masterId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addExistingMaster(masterId);
    }

    void addExistingMaster(MasterId masterId, const FtAttributes& ftAttrs) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addExistingMaster(masterId, ftAttrs);
    }

    void applyDiff(const TopologyData& fromData, const TopologyDataDiff& diff) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        topology_fixer::applyDiff(data_, fromData, diff);
    }

    void replaceEdge(EdgeId edgeId, const IdSet& edgeIds) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.replaceEdge(edgeId, edgeIds);
    }

    void addFaceEdge(DBIdType faceId, DBIdType edgeId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addFaceEdge(faceId, edgeId);
    }

    void addMasterFaceRel(
        DBIdType masterId, DBIdType faceId, FaceRelationType isInterior) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addMasterFaceRel(masterId, faceId, isInterior);
    }

    void addMasterEdgeRel(DBIdType masterId, DBIdType edgeId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.addMasterEdgeRel(masterId, edgeId);
    }

    void removeMasterFaceRel(DBIdType masterId, DBIdType faceId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeMasterFaceRel(masterId, faceId);
    }

    void removeMasterEdgeRel(DBIdType masterId, DBIdType edgeId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeMasterEdgeRel(masterId, edgeId);
    }

    void removeFaceEdgeRel(DBIdType faceId, DBIdType edgeId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeFaceEdgeRel(faceId, edgeId);
    }

    void removeMaster(DBIdType id, DeletionMode mode) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeMaster(id, mode);
    }

    void removeFace(DBIdType id, DeletionMode mode) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeFace(id, mode);
    }

    void removeEdge(DBIdType id, DeletionMode mode) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeEdge(id, mode);
    }

    void removeNode(DBIdType id) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        data_.removeNode(id);
    }

    std::vector<geolib3::Point2> faceVertices(DBIdType faceId) override
    {
        std::lock_guard<std::mutex> guard(mutex_);
        return data_.faceVertices(faceId);
    }

protected:
    std::mutex mutex_;
};

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
