#pragma once

#include <boost/optional.hpp>

#include <list>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace utils {

template <typename Id>
class UniqueList {
public:
    typedef std::list<Id> IdsList;

    typedef typename IdsList::iterator Iterator;
    typedef typename IdsList::const_iterator ConstIterator;

    UniqueList(IdsList&& ids)
        : idsList_(std::move(ids))
    {
        for (auto it = idsList_.begin(); it != idsList_.end(); ++it) {
            REQUIRE(itMap_.insert({*it, it}).second, "Value " << *it << " is already in list");
        }
    }

    UniqueList(const UniqueList&) = delete;
    UniqueList& operator = (const UniqueList&) = delete;

    UniqueList(UniqueList&& oth)
        : idsList_(std::move(oth.idsList_))
    {
        for (auto it = idsList_.begin(); it != idsList_.end(); ++it) {
            REQUIRE(itMap_.insert({*it, it}).second, "Value " << *it << " is already in list");
        }
    }

    UniqueList& operator = (UniqueList&& oth)
    {
        idsList_ = std::move(oth.idsList_);
        itMap_.clear();
        for (auto it = idsList_.begin(); it != idsList_.end(); ++it) {
            REQUIRE(itMap_.insert({*it, it}).second, "Value " << *it << " is already in list");
        }
        return *this;
    }

    size_t size() const { return idsList_.size(); }
    size_t empty() const { return idsList_.empty(); }

    Iterator begin() { return idsList_.begin(); }
    Iterator end() { return idsList_.end(); }

    ConstIterator begin() const { return idsList_.begin(); }
    ConstIterator end() const { return idsList_.end(); }

    Iterator find(Id id)
    {
        auto mapIt = itMap_.find(id);
        return mapIt == itMap_.end() ? end() : mapIt->second;
    }

    bool exists(Id id) { return find(id) != end(); }

    /**
     * Replaces id with newIds in idsList preserving order.
     * Returns iterator to next element after inserted.
     * Requires id to exist in list.
     */
    Iterator
    replace(Id id, const IdsList& newIds)
    {
        auto listIt = find(id);
        REQUIRE(listIt != end(), "Id " << id << " is not in unique list");
        if (newIds.size() == 1 && newIds.front() == id) {
            return std::next(listIt);
        }
        itMap_.erase(id);
        listIt = idsList_.erase(listIt);
        for (auto newId : newIds) {
            auto insertedIt = idsList_.insert(listIt, newId);
            REQUIRE(itMap_.insert({newId, insertedIt}).second,
                "Id " << newId << " is already in list");
        }
        return listIt;
    }

private:
    IdsList idsList_;
    std::unordered_map<Id, typename IdsList::iterator> itMap_;
};

} // namespace utils
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
