#include <mapreduce/yt/interface/client.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/geo_helpers.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/graph_load.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/hyp_converter.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/hyp_generator.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/hyp_serialize.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/consts.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/revision_roads_miner_db.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/revision_snapshot_holder.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/yt_ops.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/geom_utils.h>

#include <boost/date_time/gregorian/gregorian.hpp>
#include <iostream>
#include <sstream>

namespace mwc = maps::wiki::common;
namespace mwr = maps::wiki::revision;
namespace mwt = maps::wiki::traffic_analyzer;
namespace mg = maps::geolib3;
namespace bg = boost::gregorian;

std::string dumpHypothesesInTsv(
    const std::vector<mwt::OnewayHypothesisRevision>& hyposRev)
{
    std::ostringstream ss;
    ss << "lon\tlat\tdescription\tobject_id\n";
    for (const auto& hyp : hyposRev) {
        auto positionGeo = mg::mercator2GeoPoint(hyp.positionMerc);
        ss << std::setprecision(8) << positionGeo.x() << '\t';
        ss << std::setprecision(8) << positionGeo.y() << '\t';
        ss << "direction should be " << hyp.direction << '\t';
        ss << hyp.revId.objectId() << '\n';
    }
    return ss.str();
}

/*
    Input:
      -- tracks matching statistics (at YT) for chosen signals days
      -- static road graph, existing at local machine or at YT
    Output:
      -- to stdout: oneway hypotheses in terms of revision
         in tsv format for feedback importer
*/
int main(int argc, char* argv[]) try
{
    NYT::Initialize(argc, const_cast<const char**>(argv));

    maps::cmdline::Parser parser;

    auto revisionConfig = parser.string("revision-config")
        .help("Path to revision db configuration file")
        .required();

    auto lastSignalsDay = parser.string("last-signals-day")
        .help("Last signals day")
        .required();

    auto averagingWindow = parser.num("average-window")
        .help("Averaging window size. For example. If equal to 1 - only "
              "last day signals will be analyzed. If equal to 2 - last day "
              "and the day before last. And so on. By default is equal to 1");

    auto bboxGeoString = parser.string("bbox-geo")
        .defaultValue(mwt::WORLD_BBOX_GEO_STRING)
        .help("Comma separated string with bounding box (Geo) "
              "of hypotheses area. String format: lon1,lat1,lon2,lat2");

    auto alarmMatchRatio = parser.real("alarm-match-ratio")
        .help("Hypothesis parameter. If matched ratio of tracks for particular "
              "road is greater than this value, a hypothesis will be produced")
        .required();

    auto minBothEdgeTracks = parser.num("min-both-edge-tracks")
        .help("Hypothesis parameter. If number of matched tracks for "
              "particular road (on both directions) is less than this value, "
              "no hypothesis will be produced")
        .required();

    auto maxFunClass = parser.num("max-funclass")
        .help("Hypothesis parameter. If fun-class of particular road is "
              "greater than this value, no hypothesis will be produced")
        .required();

    parser.parse(argc, argv);

    // Take matching statistics from YT
    INFO() << "Mining matched tracks statistics for graph roads";
    int avWindowSize = 1;
    if (averagingWindow.defined()) {
        avWindowSize = averagingWindow;
    }
    bg::date_period signalsDays(
        bg::from_string(lastSignalsDay) - bg::date_duration(avWindowSize - 1),
        bg::from_string(lastSignalsDay) + bg::date_duration(1)
    );
    auto persIdsToTracks = mwt::getYTRoadsStat(signalsDays);

    // Choosing graph version
    auto graphVersion = mwt::oldestGraphVersionOfMatchedTracks(signalsDays);

    // Choose roads from bbox
    INFO() << "Loading static road graph";
    mwt::GraphMmapped graph = mwt::loadGraph(graphVersion);

    INFO() << "Loading graph roads from bounding box";
    auto bboxGeo = mwc::bboxFromCommaSeparatedCoords(bboxGeoString);
    auto twoWayRoads = graph.getTwoWayRoads(bboxGeo);

    // Produce hypotheses
    INFO() << "Generating one-way hypotheses for static road graph";
    mwt::OnewayGenParams genParams(
        alarmMatchRatio, maxFunClass, minBothEdgeTracks);
    mwt::OnewayHypothesisGenerator gen(
        graph, std::move(persIdsToTracks), genParams);

    auto hypsGraph = gen.generate(twoWayRoads);

    // Convert hypotheses
    INFO() << "Creating Revision Shapshot";
    mwc::ExtendedXmlDoc xmlDoc(revisionConfig);
    mwt::RevisionSnapshotHolder snapshotHolder(xmlDoc);

    auto revRoadsMiner =
        std::make_unique<mwt::RevisionRoadsMinerDB>(snapshotHolder.snapshot());
    mwt::OnewayHypothesisConverter converter(graph, std::move(revRoadsMiner));
    auto hypsRev = mwt::convertHypsGraphToRevision(hypsGraph, converter);

    // Dump hypotheses
    std::cout << dumpHypothesesInTsv(hypsRev);

    return 0;

} catch (const maps::Exception& e) {
    INFO() << e;
    return 0;
} catch (const std::exception& e) {
    INFO() << e.what();
    return 0;
}
