#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/geo_helpers.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/graph_load.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/hypotheses.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/hyp_serialize.h>
#include <maps/wikimap/mapspro/tools/traffic_analyzer/lib/revision_snapshot_holder.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <iostream>
#include <fstream>

namespace mwc = maps::wiki::common;
namespace mwr = maps::wiki::revision;
namespace mwt = maps::wiki::traffic_analyzer;

std::string polylineToEasyview(const maps::geolib3::Polyline2& poly)
{
    std::ostringstream stream;
    for (const auto& point : poly.points()) {
        stream << std::setprecision(8) << point.x() << " ";
        stream << std::setprecision(8) << point.y() << " ";
    }
    return stream.str();
}

/*
    Purpose of this util is to vizualize either graph hypotheses,
    either revision hypotheses, or both types hypotheses together.
    The last option is useful for hypotheses convertion between
    graph and revision.
    The result is writed to stdout in format suitable for Easyview.

    Example:
    hyp_viz --graph-version 18.01.31 --hyp-graph-file hyps_graph.txt | easyview
*/
int main(int argc, char* argv[]) try
{
    maps::cmdline::Parser parser;

    auto hypsGraphFile = parser.string("hyps-graph-file")
        .help("File with hypotheses in terms of static road graph. "
              "In json format");

    auto hypsRevisionFile = parser.string("hyps-revision-file")
        .help("File with hypotheses in terms of revision. In json format");

    auto graphVersion = parser.string("graph-version")
        .help("Graph version");

    auto revisionConfig = parser.string("revision-config")
        .help("Path to revision db configuration file");

    parser.parse(argc, argv);

    if (hypsGraphFile.defined()) {
        REQUIRE(graphVersion.defined(),
            "Graph version must be defined when visualizing graph hypotheses");

        INFO() << "Loading static road graph";
        mwt::GraphMmapped graph = mwt::loadGraph(graphVersion);

        std::ifstream hypsStream(hypsGraphFile);
        REQUIRE(hypsStream, "Unable to read file " << hypsGraphFile);

        std::cout << "!linestyle=red:8\n";

        std::string hypStr;
        while (hypsStream >> hypStr) {
            mwt::OnewayHypothesisGraph hyp =
                mwt::fromJson<mwt::OnewayHypothesisGraph>(hypStr);
            REQUIRE(hyp.roadTraffic.graphVersion == graphVersion,
                "Hypotethis graph version differs from the loaded graph version");
            std::cout << polylineToEasyview(graph.roadInfo(hyp.roadId).geomGeo);
            std::cout << "\n";
        }
    }

    if (hypsRevisionFile.defined()) {
        REQUIRE(revisionConfig.defined(), "Revision config must be defined " <<
            "when visualizing revision hypotheses");

        INFO() << "Creating Revision Shapshot";
        mwc::ExtendedXmlDoc xmlDoc(revisionConfig);
        mwt::RevisionSnapshotHolder snapshotHolder(xmlDoc);
        auto revisionReader = snapshotHolder.snapshot().reader();

        std::ifstream hypsStream(hypsRevisionFile);
        REQUIRE(hypsStream, "Unable to read file " << hypsRevisionFile);

        std::cout << "!linestyle=blue:4\n";

        std::string hypStr;
        while (hypsStream >> hypStr) {
            mwt::OnewayHypothesisRevision hyp =
                mwt::fromJson<mwt::OnewayHypothesisRevision>(hypStr);
            auto roadRev = revisionReader.loadRevision(hyp.revId);
            std::cout << polylineToEasyview(
                maps::geolib3::convertMercatorToGeodetic(
                    mwt::polylineMercFromRoadRev(roadRev)
                )
            );
            std::cout << hyp.direction << "\n";
        }
    }

    return 0;

} catch (const maps::Exception& e) {
    std::cout << e << "\n";
    return 0;
} catch (const std::exception& e) {
    std::cout << e.what() << "\n";
    return 0;
}
