#include "graph_mmapped.h"
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/jams/static_graph2/base_index.h>

#include <boost/range/algorithm_ext/erase.hpp>
#include <unordered_set>

using maps::jams::static_graph2::EdgeData;
using maps::jams::static_graph2::GraphData;
using maps::jams::static_graph2::BaseIndex;
using maps::jams::static_graph2::PersistentIndex;

namespace maps::wiki::traffic_analyzer {

namespace {

bool roadIsAvailableForTransport(uint32_t roadId, const GraphData& graphData)
{
    auto roadType = graphData.edgeData(roadId).type();
    return roadType == EdgeData::NamedRoad ||
           roadType == EdgeData::UTurn ||
           roadType == EdgeData::FrontageRoad;
}

std::vector<uint32_t>
removeUnavailableRoads(
    std::vector<uint32_t> roads, const GraphData& graphData)
{
    boost::range::remove_erase_if(roads, [&](uint32_t roadId){
        return !roadIsAvailableForTransport(roadId, graphData);
    });
    return roads;
}

/*
    Oneway road doesn't have reverse road in index.
    Twoway road has two roads in index - forward and backward edges.
    We want to take only one of them.
*/
std::vector<uint32_t>
removeOnewayAndDuplicateRoads(
    const std::vector<uint32_t>& roads, const BaseIndex& edgesIndex)
{
    std::unordered_set<uint32_t> roadsFiltered;
    for (auto roadId : roads) {
        auto roadIdRev = edgesIndex.reverse(roadId);
        if (roadIdRev && !roadsFiltered.count(*roadIdRev)) {
            roadsFiltered.insert(roadId);
        }
    }
    return std::vector<uint32_t>(roadsFiltered.begin(), roadsFiltered.end());
}

std::vector<TwoWayRoadIds> createTwoWayIdsRoads(
    const std::vector<uint32_t>& roads,
    const BaseIndex& edgesIndex,
    const PersistentIndex& persIndex)
{
    /*
        Theoretically some roads may not have persistent id.
        They are minor - skip'em.
    */
    std::vector<TwoWayRoadIds> res;
    for (auto roadId : roads) {
        auto roadIdRev = edgesIndex.reverse(roadId);
        if (roadIdRev == boost::none) {
            continue;
        }

        auto roadIdPers = persIndex.findLongId(roadId);
        auto roadIdRevPers = persIndex.findLongId(*roadIdRev);

        if (roadIdPers && roadIdRevPers) {
            res.push_back({
                roadId,
                *roadIdRev,
                roadIdPers->value(),
                roadIdRevPers->value()
            });
        }
    }
    return res;
}

} // namespace anonymous

GraphMmapped::GraphMmapped(const std::string& dataFile,
                           const std::string& edgesRtreeFile,
                           const std::string& persistentIndexFile) :
    data_(dataFile),
    rtree_(edgesRtreeFile),
    persistentIndex_(persistentIndexFile)
{
    REQUIRE(data_->version()  == rtree_->version() &&
            rtree_->version() == persistentIndex_.version(),
            "Loaded graph files have different versions");

    version_ = data_->version();
}

const std::string& GraphMmapped::version() const
{
    return version_;
}

GraphRoadInfo GraphMmapped::roadInfo(uint32_t roadId) const
{
    return {data_->edgeData(roadId).category(),
            data_->edgeData(roadId).polyline()};
}

std::vector<TwoWayRoadIds>
GraphMmapped::getTwoWayRoads(const geolib3::BoundingBox& bboxGeo) const
{
    std::vector<uint32_t> roadIds = rtree_->baseEdgesInWindow(*data_, bboxGeo);
    roadIds = removeUnavailableRoads(std::move(roadIds), *data_);
    roadIds = removeOnewayAndDuplicateRoads(roadIds, data_->edgesBaseIndex());

    return createTwoWayIdsRoads(
        roadIds, data_->edgesBaseIndex(), persistentIndex_);
}

} // namespace maps::wiki::traffic_analyzer
