#include "hyp_converter.h"
#include "geo_helpers.h"
#include <maps/libs/geolib/include/distance.h>

namespace maps::wiki::traffic_analyzer {

namespace {

RevisionRoad
nearestRevisionRoad(
    const std::vector<RevisionRoad>& roadsRevision,
    const geolib3::Point2& pointMerc)
{
    REQUIRE(!roadsRevision.empty(), "List of road revisions is empty");

    auto nearest = std::min_element(roadsRevision.begin(), roadsRevision.end(),
        [&](const RevisionRoad& lhs, const RevisionRoad& rhs){
            return geolib3::distance(polylineMidPoint(lhs.geomMerc()), pointMerc) <
                   geolib3::distance(polylineMidPoint(rhs.geomMerc()), pointMerc);
        }
    );

    return *nearest;
}

} // namespace anonymous

OnewayHypothesisConverter::OnewayHypothesisConverter(
    const IGraph& graph, std::unique_ptr<IRevisionRoadsMiner> revRoadsMiner) :
        graph_(graph), revRoadsMiner_(std::move(revRoadsMiner))
{
}

/*
    Convertion steps:
    1) Construct bounding box of graph road
    2) Get revision roads in this bounding box
    3) Get the nearest revision road to graph road,
       based on polylines midpoints distance.
    4) Construct global direction (vector, connecting start and end of polyline)
       for graph and revision roads polylines.
    5) Calculate sign of scalar product of global directions in order to obtain
       hypothesis direction w.r.t. revision road initial direction.
*/
boost::optional<OnewayHypothesisRevision>
OnewayHypothesisConverter::convert(const OnewayHypothesisGraph& graphHyp) const
{
    REQUIRE(graphHyp.roadTraffic.graphVersion == graph_.version(),
            "Versions of graph and hypothesis must match");

    auto polylineGraphMerc = geolib3::convertGeodeticToMercator(
        graph_.roadInfo(graphHyp.roadId).geomGeo);
    auto midPointGraphMerc = polylineMidPoint(polylineGraphMerc);

    auto roadsRevisions = revRoadsMiner_->getRoadsFromBox(
        polylineGraphMerc.boundingBox());
    if (roadsRevisions.empty()) {
        return boost::none;
    }

    auto roadRev = nearestRevisionRoad(roadsRevisions, midPointGraphMerc);
    const auto& polylineRevMerc = roadRev.geomMerc();

    auto directionRevMerc   = polylineGlobalDirection(polylineRevMerc);
    auto directionGraphMerc = polylineGlobalDirection(polylineGraphMerc);
    auto hypRevDir = (geolib3::innerProduct(directionRevMerc, directionGraphMerc) > 0) ?
        ymapsdf::rd::Direction::Forward : ymapsdf::rd::Direction::Backward;

    auto midPointRevMerc = polylineMidPoint(polylineRevMerc);

    return OnewayHypothesisRevision{
        midPointRevMerc, roadRev.id(), hypRevDir, graphHyp.roadTraffic};
}

std::vector<OnewayHypothesisRevision>
convertHypsGraphToRevision(
    const std::vector<OnewayHypothesisGraph>& hypsGraph,
    const OnewayHypothesisConverter& conv)
{
    std::vector<OnewayHypothesisRevision> hypsRev;
    for (const auto& hypGraph : hypsGraph) {
        auto hypRev = conv.convert(hypGraph);
        if (hypRev) {
            hypsRev.push_back(std::move(*hypRev));
        }
    }
    return hypsRev;
}

} // namespace maps::wiki::traffic_analyzer
