#include "hyp_serialize.h"
#include "hypotheses.h"

#include <maps/libs/json/include/value.h>

#include <boost/lexical_cast.hpp>

namespace bg = boost::gregorian;

namespace maps::wiki::traffic_analyzer {

namespace {

std::string toString(const json::Value& value)
{
    json::Builder builder;
    json::json(value, json::VariantBuilder(&builder));
    return builder;
}

} // namespace anonymous

void json(const wiki::traffic_analyzer::RoadTraffic& roadTraf,
          json::ObjectBuilder builder)
{
    builder["bothEdgeTracks"] = roadTraf.bothEdgeTracks;
    builder["matchRatio"] = roadTraf.matchRatio;
    builder["graphVersion"] = roadTraf.graphVersion;
    builder["dayBegin"] = bg::to_iso_extended_string(roadTraf.days.begin());
    builder["dayEnd"] = bg::to_iso_extended_string(roadTraf.days.end());
}

std::string toJson(const RoadTraffic& roadTraffic)
{
    return (json::Builder() << roadTraffic).str();
}

std::string toJson(const OnewayHypothesisGraph& hyp)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["roadId"] = hyp.roadId;

        builder["roadTraffic"] << [&](json::ObjectBuilder builder) {
            json(hyp.roadTraffic, builder);
        };
    };
    return builder.str();
}

std::string toJson(const OnewayHypothesisRevision& hyp)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["positionMerc"] << [&](json::ObjectBuilder builder) {
            builder["x"] = hyp.positionMerc.x();
            builder["y"] = hyp.positionMerc.y();
        };

        builder["revId"] << [&](json::ObjectBuilder builder) {
            builder["objectId"] = hyp.revId.objectId();
            builder["commitId"] = hyp.revId.commitId();
        };

        builder["direction"] = boost::lexical_cast<std::string>(hyp.direction);

        builder["roadTraffic"] << [&](json::ObjectBuilder builder) {
            json(hyp.roadTraffic, builder);
        };
    };
    return builder.str();
}

template<>
RoadTraffic fromJson(const std::string& json)
{
    json::Value value = json::Value::fromString(json);

    RoadTraffic roadTraffic(
        value["bothEdgeTracks"].as<uint32_t>(),
        value["matchRatio"].as<double>(),
        value["graphVersion"].as<std::string>(),
        bg::date_period(
            bg::from_string(value["dayBegin"].as<std::string>()),
            bg::from_string(value["dayEnd"].as<std::string>())
        )
    );

    return roadTraffic;
}

template<>
OnewayHypothesisGraph fromJson(const std::string& json)
{
    json::Value value = json::Value::fromString(json);

    OnewayHypothesisGraph hyp(
        value["roadId"].as<uint32_t>(),
        fromJson<RoadTraffic>(toString(value["roadTraffic"]))
    );

    return hyp;
}

template<>
OnewayHypothesisRevision fromJson(const std::string& json)
{
    json::Value value = json::Value::fromString(json);

    OnewayHypothesisRevision hyp(
        value["positionMerc"].construct<geolib3::Point2>("x", "y"),
        value["revId"].construct<revision::RevisionID>("objectId", "commitId"),
        boost::lexical_cast<ymapsdf::rd::Direction>(
            value["direction"].as<std::string>()),
        fromJson<RoadTraffic>(toString(value["roadTraffic"]))
    );

    return hyp;
}

} // namespace maps::wiki::traffic_analyzer
