#include "revision_roads_miner_db.h"
#include "geo_helpers.h"
#include <yandex/maps/wiki/common/rd/access_id.h>

#include <boost/lexical_cast.hpp>

namespace maps::wiki::traffic_analyzer {

namespace {

const std::string CAT_RD_EL = "cat:rd_el";
const std::string RD_EL_ACCESS_ID = "rd_el:access_id";

std::list<revision::ObjectRevision>
revisionRoadObjectsInBoundingBox(
    const revision::Snapshot& revSnapshot,
    const geolib3::BoundingBox& bboxMerc)
{
    namespace rf = revision::filters;
    const common::AccessId VEHICLES =
        common::AccessId::Bus | common::AccessId::Truck | common::AccessId::Car;

    auto geomFilter = rf::GeomFilterExpr(
        rf::GeomFilterExpr::Operation::IntersectsLinestrings,
        bboxMerc.minX(), bboxMerc.minY(), bboxMerc.maxX(), bboxMerc.maxY());

    rf::ProxyFilterExpr roadsFilter =
        rf::ObjRevAttr::isNotDeleted() &&
        rf::ObjRevAttr::isNotRelation() &&
        rf::Geom::defined() &&
        rf::Attr(CAT_RD_EL).defined();

    auto filter = roadsFilter && geomFilter;
    auto roadsRevisions = revSnapshot.objectRevisionsByFilter(filter);

    roadsRevisions.remove_if([](const revision::ObjectRevision& rev){
        const auto& attrs = rev.data().attributes;
        if (!attrs || !attrs->count(RD_EL_ACCESS_ID) || !rev.data().geometry) {
            return true;
        }

        // ignore roads unavailable for transport
        auto accessId = static_cast<common::AccessId>(
            boost::lexical_cast<int>(attrs->at(RD_EL_ACCESS_ID)));
        return !(VEHICLES & accessId);
    });

    return roadsRevisions;
}

} // namespace anonymous

RevisionRoadsMinerDB::RevisionRoadsMinerDB(revision::Snapshot snapshot) :
    revisionSnapshot_(std::move(snapshot))
{
}

std::vector<RevisionRoad>
RevisionRoadsMinerDB::getRoadsFromBox(const geolib3::BoundingBox& boxMerc) const
{
    auto revRoadObjects = revisionRoadObjectsInBoundingBox(
        revisionSnapshot_, boxMerc);

    std::vector<RevisionRoad> revRoads;
    for (const revision::ObjectRevision& revRoadObject : revRoadObjects) {
        revRoads.emplace_back(
            revRoadObject.id(),
            polylineMercFromRoadRev(revRoadObject)
        );
    }

    return revRoads;
}

} // namespace maps::wiki::traffic_analyzer
