#include "helpers.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <atomic>
#include <future>
#include <memory>
#include <chrono>

namespace maps {
namespace wiki {
namespace isocode {

namespace {

std::string
signatureToName(const std::string& funcSignature)
{
    const auto roundBracket = funcSignature.find("(");
    const auto doubleColon = funcSignature.rfind("::",roundBracket);
    size_t start = doubleColon == std::string::npos
        ? 0
        : doubleColon + 2;
    size_t length = roundBracket == std::string::npos
        ? std::string::npos
        : roundBracket - start;

    return funcSignature.substr(start, length);
}

} // namespace

pqxx::result
exec(
    pqxx::transaction_base& work,
    const std::string& query,
    const std::string& funcSignature)
{
    const auto funcName = signatureToName(funcSignature);
    INFO() << "start " << funcName;

    const auto start = std::chrono::system_clock::now();

    DEBUG() << funcName << ":" << query;

    const auto res = work.exec(query);

    const auto end = std::chrono::system_clock::now();

    const auto elapsedSeconds = std::chrono::duration_cast<std::chrono::seconds>(end - start);

    INFO() << funcName << ": "
        << elapsedSeconds.count() << " sec, "
        << res.affected_rows() << " rows.";

    return res;
}

void
safeRunner(
    std::function<void()> f,
    const std::string& name,
    std::shared_ptr<std::atomic<bool>> fail)
{
    if (*fail) {
        return;
    }

    INFO() << name << " started";
    try {
        f();
        INFO() << name << " finished";
    } catch (const maps::RuntimeError& rtErr) {
        ERROR() << name << " failed: " << rtErr;
        *fail = true;
    } catch (std::exception& ex) {
        ERROR() << name << " failed: " << ex.what();
        *fail = true;
    } catch (...) {
        ERROR() << name << " failed: Unknown exception";
        *fail = true;
    }
}

} // namespace isocode
} // namespace wiki
} // namespace maps
