#include "isocode/builder.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/lexical_cast.hpp>
#include <boost/program_options.hpp>

#include <iostream>
#include <thread>

namespace po = boost::program_options;

namespace {

const std::string OPT_HELP = "help";
const std::string OPT_CONN = "conn";
const std::string OPT_SCHEMANAME = "schemaname";
const std::string OPT_COVERAGE_DIR = "coverage-dir";
const std::string OPT_THREADS = "threads";

const int EXITCODE_OK = 0;
const int EXITCODE_LOGIC_ERROR = 1;
const int EXITCODE_RUNTIME_ERROR = 2;

template <typename Type>
Type
getValue(const po::variables_map& vm, const std::string& option)
{
    if (!vm.count(option)) {
        return {};
    }
    try {
        return vm[option].as<Type>();
    } catch (...) {
        ERROR() << "invalid type of option: " << option;
        throw;
    }
}

std::string
getNonEmptyString(const po::variables_map& vm, const std::string& option)
{
    auto value = getValue<std::string>(vm, option);
    REQUIRE(!value.empty(), "empty value of option: " << option);
    return value;
}

size_t
threadsCountDefault()
{
    auto threads = std::thread::hardware_concurrency();
    return std::max(1UL, static_cast<size_t>((threads * 2) / 3));
}

} // namespace

int main(int argc, char** argv)
{
    po::options_description desc("Usage: <tool> <options>\nOptions");
    desc.add_options()
        (OPT_HELP.c_str(),
            "show help message")
        (OPT_CONN.c_str(), po::value<std::string>(),
            "connection string)")
        (OPT_SCHEMANAME.c_str(), po::value<std::string>(),
            "ymapsdf schema name")
        (OPT_COVERAGE_DIR.c_str(), po::value<std::string>()->default_value("./"),
            "directory for file ymapsdf.mms.1")
        (OPT_THREADS.c_str(), po::value<size_t>()->default_value(threadsCountDefault()),
            "threads")
    ;

    po::variables_map vm;
    try {
        po::store(po::parse_command_line(argc, argv, desc), vm);
    } catch (po::error& e) {
        std::cerr << "error parsing command line: " << e.what() << std::endl;
        return EXITCODE_LOGIC_ERROR;
    }
    po::notify(vm);

    if (argc == 1 || vm.count(OPT_HELP)) {
        std::cerr << desc << std::endl;
        return EXITCODE_LOGIC_ERROR;
    }

    try {
        if (maps::wiki::isocode::build(
                getNonEmptyString(vm, OPT_CONN),
                getNonEmptyString(vm, OPT_SCHEMANAME),
                getNonEmptyString(vm, OPT_COVERAGE_DIR),
                getValue<size_t>(vm, OPT_THREADS))) {
            INFO() << "OK";
            return EXITCODE_OK;
        }
        ERROR() << "Fail";
    } catch(maps::Exception& ex) {
        ERROR() << ex;
    } catch(std::exception& ex) {
        ERROR() << ex.what();
    }
    return EXITCODE_RUNTIME_ERROR;
}
