#include <maps/libs/common/include/profiletimer.h>
#include <yandex/maps/coverage5/coverage.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>

#include <boost/optional.hpp>
#include <boost/none.hpp>
#include <pqxx>

#include <functional>
#include <iostream>
#include <sstream>
#include <string>
#include <thread>
#include <vector>


namespace {
const int NO_REGION_ID = 0;

// const std::string LAYERS_FILE = "/usr/share/yandex/maps/coverage5/geoid.mms.1";
// const std::string LAYER_NAME = "geoid";
const std::string LAYERS_FILE = "/home/gagauzev/tmp/rusb.mms.1";
const std::string LAYER_NAME = "rus";
}

using BoundingBox = maps::geolib3::BoundingBox;
using Points = std::vector<maps::geolib3::Point2>;
using Threads = std::vector<std::thread>;
using RegionsToPointsNumber = std::map<maps::coverage5::RegionId, size_t>;


class DB {
public:
    DB():
        conn_(CONNECTION_PARAMS)
    {
        std::cout << "Schema: " << SCHEMA << std::endl;
    }

    const Points& loadPoints(size_t pointsNumber) {
        std::cout << "Number of points to be loaded: " << pointsNumber << std::endl;

        ProfileTimer loadTimer;
        pqxx::work txn(conn_);
        for (const auto& row: txn.exec(query(pointsNumber))) {
            maps::geolib3::Point2 point{
                row["lon"].as<double>(),
                row["lat"].as<double>()
            };
            points_.push_back(std::move(point));
        }
        std::cout << "Load time: " << loadTimer << std::endl;
        std::cout << "Number of loaded points: " << points_.size() << std::endl;

        return points_;
    }

private:
    const std::string CONNECTION_PARAMS =
        "host=garden-pgm-rus01h.tst.maps.yandex.ru "
        "port=5432 "
        "dbname=garden "
        "user=mapsgarden "
        "password=gardenTst";
    const std::string SCHEMA = "ymapsdf_beta_yandex_russia_20170714_123165_518_52410114";

    pqxx::connection conn_;
    Points points_;

    std::string query(size_t pointsNumber) const {
        std::stringstream result;

        // result
        //     << "SELECT"
        //     << "  ST_X(shape) as lon,"
        //     << "  ST_Y(shape) as lat "
        //     << "FROM " << SCHEMA << ".node "
        //     << "LIMIT " << pointsNumber << ";";

        // Russia boundaries
        result
            << "SELECT"
            << "  ST_X(((ST_DumpPoints(shape)).geom)) as lon,"
            << "  ST_Y(((ST_DumpPoints(shape)).geom)) as lat "
            << "FROM " << SCHEMA << ".ad "
            << "JOIN " << SCHEMA << ".ad_geom USING(ad_id) "
            << "WHERE level_kind IN (1)"
            << "  AND g_ad_id IS NULL"
            << "  AND isocode = 'RU' "
            << "LIMIT " << pointsNumber;

        return result.str();
    }
};


void measureCoverageTime(
    size_t offset, size_t count, const Points& points,
    const maps::coverage5::Layer& layer, RegionsToPointsNumber &regionsToPointsNumber)
{
    auto id = std::this_thread::get_id();

    ProfileTimer coverageTimer;
    for (size_t pointsIx{offset}; pointsIx < offset + count; ++pointsIx) {
        auto regions = layer.regions(points[pointsIx], boost::none);

        if (!regions.empty()) {
            for (const auto& region: regions) {
                if (region.id()) {
                    regionsToPointsNumber[*region.id()]++;
                } else {
                    regionsToPointsNumber[NO_REGION_ID]++;
                }
            }
        } else {
            regionsToPointsNumber[NO_REGION_ID]++;
        }
    }
    std::cout << "Thread's " << id
              << " coverage time for the chunk[" << offset << "; " << offset + count << "): " << coverageTimer << std::endl;
}


void regionsToPointsStat(const std::vector<RegionsToPointsNumber>& regionsToPointsNumbers)
{
    RegionsToPointsNumber totalRegionsToPointsNumber;
    size_t totalRelationsNumber{0};
    for (const auto& regionsToPointsNumber: regionsToPointsNumbers) {
        for (const auto& regionToPointsNumber: regionsToPointsNumber) {
            totalRegionsToPointsNumber[regionToPointsNumber.first] += regionToPointsNumber.second;
            totalRelationsNumber += regionToPointsNumber.second;
        }
    }
    for (const auto& regionToPointsNumber: totalRegionsToPointsNumber) {
        std::cout << "Region " << regionToPointsNumber.first
                  << " contains " << regionToPointsNumber.second << " points." << std::endl;
    }
    std::cout << "Total regions to points number: " << totalRelationsNumber << std::endl;
}


int main()
{
    size_t pointsNumber = 100000000;
    const size_t threadsNumber = 1;

    maps::coverage5::Coverage coverage(LAYERS_FILE);
    auto& layer = coverage[LAYER_NAME];

    DB db;

    std::cout << "Layers file: " << LAYERS_FILE << std::endl
              << "Layer name: " << LAYER_NAME << std::endl
              << "Number of threads: " << threadsNumber << std::endl
              << "Hardware concurrency: " << std::thread::hardware_concurrency() << std::endl;

    const auto& points = db.loadPoints(pointsNumber);
    pointsNumber = points.size();
    const size_t chunkSize = pointsNumber / threadsNumber;
    std::cout << "Number of points in a chunk: " << chunkSize << std::endl;

    ProfileTimer totalTimer;
    Threads threads;
    std::vector<RegionsToPointsNumber> regionsToPointsNumbers(threadsNumber);

    for (size_t threadIx{0}; threadIx < threadsNumber; ++threadIx) {
        threads.push_back(std::thread{
            measureCoverageTime,
            threadIx * chunkSize,
            chunkSize,
                std::cref(points), std::cref(layer),
                std::ref(regionsToPointsNumbers[threadIx])});
    }
    for (auto &thread: threads) {
        thread.join();
    }
    std::cout << "Total coverage time: " << totalTimer << std::endl;

    regionsToPointsStat(regionsToPointsNumbers);
}
