#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/file_utils.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/coverage5/coverage.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/concurrent/include/threadpool.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <boost/lexical_cast.hpp>
#include <boost/program_options.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>

#include <pqxx/pqxx>
#include <cstdint>
#include <iostream>
#include <unordered_set>
#include <vector>

namespace po = boost::program_options;
namespace fs = boost::filesystem;

namespace maps {
namespace wiki {
namespace {

typedef int64_t ID;
typedef std::unordered_set<std::string> Isocodes;

const std::string OPT_HELP = "help";
const std::string OPT_CONN = "conn";
const std::string OPT_SCHEMANAME = "schemaname";
const std::string OPT_FILE = "file";
const std::string OPT_THREADS = "threads";
const std::string OPT_CHECK_EXISTS = "check-exists";
const std::string OPT_TRUNCATE = "truncate";

const int EXITCODE_OK = 0;
const int EXITCODE_LOGIC_ERROR = 1;
const int EXITCODE_RUNTIME_ERROR = 2;

const std::string UNKNOWN_ISOCODE_001 = "001";
const size_t IDS_BATCH_SIZE = 10000;
const std::string YMAPSDF_EXPORT_EXTENSIONS =
    "/usr/share/yandex/maps/garden/ymapsdf2/extensions/export/";

template <typename Type>
Type
getValue(const po::variables_map& vm, const std::string& option)
{
    if (!vm.count(option)) {
        return {};
    }
    try {
        return vm[option].as<Type>();
    } catch (...) {
        ERROR() << "invalid type of option: " << option;
        throw;
    }
}

std::string
getNonEmptyString(const po::variables_map& vm, const std::string& option)
{
    auto value = getValue<std::string>(vm, option);
    REQUIRE(!value.empty(), "empty value of option: " << option);
    return value;
}


struct Options
{
    explicit Options(const po::variables_map& vm)
        : schemaName(getNonEmptyString(vm, OPT_SCHEMANAME))
        , connStr(getNonEmptyString(vm, OPT_CONN))
        , coverageFile(getNonEmptyString(vm, OPT_FILE))
        , threads(getValue<size_t>(vm, OPT_THREADS))
        , checkExists(vm.count(OPT_CHECK_EXISTS))
        , isTruncate(vm.count(OPT_TRUNCATE))
    {
        ASSERT(threads);
        ASSERT(!checkExists || !isTruncate);
    }

    const std::string schemaName;
    const std::string connStr;
    const std::string coverageFile;
    const size_t threads;
    const bool checkExists;
    const bool isTruncate;
};

class Runner : private Options
{
public:
    explicit Runner(const po::variables_map& vm)
        : Options(vm)
        , coverage_(coverageFile)
        , layer_(coverage_[coverage_.layersNames().front()])
    {
        checkRegions();
    }

    int run() const
    {
        buildTables();

        createIsocodes<geolib3::Point2>("rd_jc", "rd_jc_id");
        createIsocodes<geolib3::Point2>("node", "node_id");
        createIsocodes<geolib3::Polyline2>("rd_el", "rd_el_id");
        createIsocodes<geolib3::Polyline2>("edge", "edge_id");

        return EXITCODE_OK;
    }


private:
    template <typename GeomType>
    void createIsocodes(
        const std::string& tableName,
        const std::string& idName) const
    {
        INFO() << tableName << " start";

        concurrent::Params params;
        params.setThreadsNumber(threads);
        concurrent::ThreadPool pool(params);

        createIsocodes<GeomType>(tableName, idName, pool);
        pool.join();

        INFO() << tableName << " processed";
    }

    template <typename GeomType>
    void createIsocodes(
        const std::string& tableName,
        const std::string& idName,
        concurrent::ThreadPool& pool) const
    {
        pqxx::connection conn(connStr);
        pqxx::work work(conn);

        std::unordered_set<ID> existedIds;
        if (checkExists) {
            INFO() << tableName << " loading existing data";
            for (const auto& row : work.exec(
                    "SELECT " + idName +
                    " FROM " + schemaName + "." + tableName + "_isocode")) {
                existedIds.insert(row[0].as<ID>());
            }
            INFO() << tableName << " existing isocodes, ids size: "  << existedIds.size();
        }

        ID lastId = 0;
        for (;;) {
            std::vector<ID> ids;
            std::ostringstream query;
            query << "SELECT " << idName << " FROM " << schemaName << "." << tableName;
            query << " WHERE " << idName << " > " << lastId;
            query << " ORDER BY 1 LIMIT " << IDS_BATCH_SIZE;
            auto rows = work.exec(query.str());
            if (rows.empty()) {
                break;
            }
            ids.reserve(rows.size());
            for (const auto& row : rows) {
                lastId = row[0].as<ID>();
                if (!existedIds.count(lastId)) {
                    ids.push_back(lastId);
                }
            }

            pool.add([=](){ batch<GeomType>(tableName, idName, ids); });

            if (rows.size() < IDS_BATCH_SIZE) {
                break;
            }
        }
    }

    template <typename GeomType>
    void batch(
        const std::string& tableName,
        const std::string& idName,
        const std::vector<ID>& ids) const
    {
        if (ids.empty()) {
            return;
        }
        auto batchInfo = "batch " + tableName +
            " [" + std::to_string(ids[0]) + "," +
                   std::to_string(ids.back()) + "]";
        INFO() << batchInfo << " loading";

        pqxx::connection conn(connStr);
        pqxx::work work(conn);

        auto rows = work.exec(
            "SELECT " + idName + ", ST_AsBinary(shape)"
            " FROM " + schemaName + "." + tableName +
            " WHERE " + idName + " IN (" + common::join(ids, ',') + ")");
        ASSERT(rows.size() == ids.size());

        INFO() << batchInfo << " calculating";

        std::list<std::string> values;
        for (const auto& row : rows) {
            auto id = row[0].as<std::string>();
            auto wkb = pqxx::binarystring(row[1]).str();

            for (const auto& isocode : calcIsocodes<GeomType>(id, wkb)) {
                values.push_back(id + ",'" + isocode + "'");
            }
        }
        if (!values.empty()) {
            INFO() << batchInfo << " writing";
            work.exec(
                "INSERT INTO " + schemaName + "." + tableName + "_isocode"
                " VALUES (" + common::join(values, "),(") + ")");
            work.commit();
        }
        INFO() << batchInfo << " done";
    }

    void checkRegions() const
    {
        pqxx::connection conn(connStr);
        pqxx::work work(conn);

        Isocodes isocodes;

        auto query =
            "SELECT isocode FROM " + schemaName + ".ad "
            " WHERE level_kind=1 AND g_ad_id IS NULL AND isocode IS NOT NULL";
        for (const auto& row : work.exec(query)) {
            isocodes.insert(row[0].as<std::string>());
        }
        INFO() << "ad isocodes: " << isocodes.size();
        auto regions = layer_.regions(boost::none);
        INFO() << "layer regions: " << regions.size();
        for (const auto& region : regions) {
            auto regionId = region.id();
            ASSERT(regionId);
            std::string isocode = region.metaData();
            REQUIRE(isocodes.count(isocode),
                "unknown isocode for layer region: " << *regionId << " : " << isocode);
        }
    }

    void buildTables() const
    {
        pqxx::connection conn(connStr);
        pqxx::work work(conn);
        work.exec("SET search_path=" + schemaName + ",public");

        std::set<std::string> tables;
        for (const auto& row : work.exec(
                "SELECT tablename FROM pg_tables"
                " WHERE schemaname='" + schemaName + "'")) {
            tables.insert(row[0].as<std::string>());
        }

        fs::path path(YMAPSDF_EXPORT_EXTENSIONS);
        for(fs::directory_iterator itr(path), end; itr != end; ++itr) {
            auto name = itr->path().filename().string();
            auto pos = name.find("_isocode.sql");
            if (pos == std::string::npos) {
                continue;
            }
            auto tablename = name.substr(0, pos + 8);
            if (tables.count(tablename)) {
                INFO() << "Table " << tablename << " already exists";
                if (isTruncate) {
                    work.exec("TRUNCATE " + tablename);
                    INFO() << "Table " << tablename << " truncated";
                }
            } else {
                auto filename = itr->path().string();
                INFO() << "Creating table " << tablename;
                auto data = maps::common::readBinaryFile(filename);
                boost::replace_all(data, "{_self}", tablename);
                work.exec(data);
            }
        }
        work.commit();
    }

    void addIsocodes(
        const std::string& id,
        const geolib3::Point2& point,
        Isocodes& isocodes) const
    {
        auto regions = layer_.regions(point, boost::none);
        if (regions.empty()) {
            if (isocodes.insert(UNKNOWN_ISOCODE_001).second) {
                WARN() << "no regions for " << id;
            }
            return;
        }
        for (const auto& region : regions) {
            isocodes.insert(region.metaData());
        }
    }

    template <typename GeomType>
    Isocodes calcIsocodes(
        const std::string& id,
        const std::string& wkb) const;

private:
    const coverage5::Coverage coverage_;
    const coverage5::Layer& layer_;
};

template <>
Isocodes
Runner::calcIsocodes<geolib3::Point2>(const std::string& id, const std::string& wkb) const
{
    std::istringstream rawStream(wkb);
    auto geom = geolib3::WKB::read<geolib3::Point2>(rawStream);

    std::unordered_set<std::string> isocodes;
    addIsocodes(id, geom, isocodes);
    return isocodes;
}

template <>
Isocodes
Runner::calcIsocodes<geolib3::Polyline2>(const std::string& id, const std::string& wkb) const
{
    std::istringstream rawStream(wkb);
    auto geom = geolib3::WKB::read<geolib3::Polyline2>(rawStream);

    std::unordered_set<std::string> isocodes;
    for (const auto& point : geom.points()) {
        addIsocodes(id, point, isocodes);
    }
    return isocodes;
}

} // namespace
} // namespace wiki
} // namespace maps

int main(int argc, char** argv)
{
    using namespace maps::wiki;

    po::options_description desc("Usage: <tool> <options>\nOptions");
    desc.add_options()
        (OPT_HELP.c_str(),
            "show help message")
        (OPT_CONN.c_str(), po::value<std::string>(),
            "connection string)")
        (OPT_SCHEMANAME.c_str(), po::value<std::string>(),
            "ymapsdf schema name")
        (OPT_FILE.c_str(), po::value<std::string>(),
            "coverage file path")
        (OPT_THREADS.c_str(), po::value<size_t>()->default_value(16),
            "threads")
        (OPT_CHECK_EXISTS.c_str(),
            "check existing data")
        (OPT_TRUNCATE.c_str(),
            "truncate isocode tables")
    ;

    po::variables_map vm;
    try {
        po::store(po::parse_command_line(argc, argv, desc), vm);
    } catch (po::error& e) {
        std::cerr << "error parsing command line: " << e.what() << std::endl;
        return EXITCODE_LOGIC_ERROR;
    }
    po::notify(vm);

    if (argc == 1 || vm.count(OPT_HELP)) {
        std::cerr << desc << std::endl;
        return EXITCODE_LOGIC_ERROR;
    }

    try {
        return Runner(vm).run();
    } catch(maps::Exception& ex) {
        ERROR() << ex;
    } catch(std::exception& ex) {
        ERROR() << ex.what();
    }
    return EXITCODE_RUNTIME_ERROR;
}
