/* Collect operating transport systems */
CREATE TEMPORARY TABLE operating_transport_system AS
SELECT ft_id FROM ft
WHERE ft_type_id = 631 /* transport-system */
EXCEPT
SELECT ft_id FROM ft_attr WHERE key = 'not_operating';


/* Collect operating transport routes */
CREATE TEMPORARY TABLE operating_transport_route AS
SELECT ft.ft_id FROM ft
JOIN operating_transport_system ON (p_ft_id = operating_transport_system.ft_id)
WHERE ft_type_id IN
(
    2201, /* transport-bus-route */
    2202, /* transport-trolleybus-route */
    2203, /* transport-tram-route */
    2204, /* transport-minibus-route */
    625,  /* transport-metro-light-line */
    626,  /* transport-metro-tram-line */
    627,  /* transport-metro-monorail-line */
    628,  /* transport-metro-bus-line */
    629,  /* transport-metro-cable-line */
    630,  /* transport-metro-funicular-line */
    632,  /* transport-metro-line */
    2208  /* transport-waterway-route */
)
EXCEPT
SELECT ft_id FROM ft_attr WHERE key = 'not_operating';


/* Collect published transport routes */
CREATE TEMPORARY TABLE published_transport_route AS
SELECT ft_id FROM operating_transport_route
UNION
SELECT ft_id FROM ft
WHERE p_ft_id IS NULL
AND ft_type_id IN
(
    625, /* transport-metro-light-line */
    626, /* transport-metro-tram-line */
    627, /* transport-metro-monorail-line */
    628, /* transport-metro-bus-line */
    629, /* transport-metro-cable-line */
    630, /* transport-metro-funicular-line */
    632  /* transport-metro-line */
)
EXCEPT
SELECT ft_id FROM ft_attr WHERE key = 'not_operating';


/* Collect not-operating transport metro routes */
CREATE TEMPORARY TABLE not_published_transport_metro_route AS
SELECT ft_id FROM ft
WHERE ft_type_id IN
(
    625, /* transport-metro-light-line */
    626, /* transport-metro-tram-line */
    627, /* transport-metro-monorail-line */
    628, /* transport-metro-bus-line */
    629, /* transport-metro-cable-line */
    630, /* transport-metro-funicular-line */
    632  /* transport-metro-line */
)
EXCEPT
SELECT ft_id FROM published_transport_route;

/* Collect operating transport threads */
CREATE TEMPORARY TABLE operating_transport_thread AS
SELECT ft.ft_id FROM ft
JOIN operating_transport_route ON (p_ft_id = operating_transport_route.ft_id)
WHERE ft_type_id IN
(
    2205, /* transport-bus-thread */
    2212, /* transport-tram-thread */
    2213, /* transport-waterway-thread */
    2214  /* transport-metro-thread */
)
EXCEPT
SELECT ft_id FROM ft_attr WHERE key = 'not_operating';

/* Collect operating transport thread_stops */
CREATE TEMPORARY TABLE operating_transport_thread_stop AS
SELECT ft.ft_id FROM ft
JOIN operating_transport_thread ON (p_ft_id = operating_transport_thread.ft_id)
WHERE ft_type_id = 2206 /* transport-thread-stop */;

/* Collect stops without operating thread */
CREATE TEMPORARY TABLE transport_stop_without_operating_thread AS
SELECT ft_id FROM ft
WHERE ft_type_id IN
(
    671, /* transport-bus-stop */
    602, /* transport-waterway-wharf */
    603, /* transport-waterway-seaport */
    604  /* transport-waterway-riverport */
)
EXCEPT
SELECT master_ft_id FROM ft_ft
JOIN operating_transport_thread_stop ON (slave_ft_id = ft_id)
WHERE role = 'assigned_thread_stop';

/* Collect stations without operating thread */
CREATE TEMPORARY TABLE transport_station_without_operating_thread AS
SELECT ft_id FROM ft
WHERE ft_type_id = 633 /* transport-metro-station */
EXCEPT
SELECT master_ft_id FROM ft_ft
JOIN operating_transport_thread_stop ON (slave_ft_id = ft_id)
WHERE role = 'assigned_thread_stop';

/* Collect bus stops without operating route */
CREATE TEMPORARY TABLE transport_bus_stop_without_route AS
SELECT ft_id FROM ft WHERE ft_type_id = 671 /* transport-bus-stop */
EXCEPT
SELECT slave_ft_id FROM ft_ft
JOIN operating_transport_route ON (master_ft_id = ft_id)
WHERE role = 'assigned';

/* Collect metro stations without published route */
CREATE TEMPORARY TABLE transport_metro_stations_without_published_route AS
SELECT ft.ft_id
FROM ft
JOIN not_published_transport_metro_route ON (p_ft_id = not_published_transport_metro_route.ft_id)
WHERE ft_type_id = 633 /* transport-metro-station */
UNION
SELECT ft_id
FROM ft
WHERE ft_type_id = 633 /* transport-metro-station */
AND p_ft_id IS NULL;

/* Drop import_source for stops without operating thread */
DELETE FROM ft_source
USING transport_stop_without_operating_thread
WHERE source_type_id = 3 /* mtr */
AND ft_source.ft_id = transport_stop_without_operating_thread.ft_id;

/* Drop trivial generated import_source for stations without operating thread */
DELETE FROM ft_source
USING transport_station_without_operating_thread
WHERE source_type_id = 3 /* mtr */
AND ft_source.ft_id = transport_station_without_operating_thread.ft_id
AND ft_source.source_id = ft_source.ft_id::text;

/* Drop trivial generated import_source for exits of stations without thread */
DELETE FROM ft_source
USING ft, transport_station_without_operating_thread
WHERE source_type_id = 3 /* mtr */
AND ft_source.ft_id = ft.ft_id
AND ft_type_id = 634 /* transport-metro-exit */
AND p_ft_id = transport_station_without_operating_thread.ft_id
AND ft_source.source_id = ft_source.ft_id::text;


CREATE TEMPORARY TABLE deleted_ft
(
    ft_id bigint NOT NULL
);

/* Collect to delete metro_stations without published parent route */
INSERT INTO deleted_ft
SELECT ft_id FROM transport_metro_stations_without_published_route;

/* Collect to delete exits of stations without published parent route */
INSERT INTO deleted_ft
SELECT ft.ft_id
FROM ft
JOIN transport_metro_stations_without_published_route
    ON (ft.p_ft_id = transport_metro_stations_without_published_route.ft_id)
WHERE ft_type_id = 634 /* transport-metro-exit */;

/* Collect to delete not-operating transport_system */
INSERT INTO deleted_ft
SELECT ft_id
FROM ft
JOIN ft_attr USING (ft_id)
WHERE key = 'not_operating'
AND ft_type_id = 631 /* transport-system */;

/* Clear p_ft_id for metro lines in not-operating transport systems */
UPDATE ft
SET p_ft_id = NULL
WHERE p_ft_id IN (
    SELECT ft_id
    FROM ft
    JOIN ft_attr USING (ft_id)
    WHERE key = 'not_operating'
    AND ft_type_id = 631 /* transport-system */
);

/* Collect to delete thread_stops, threads, routes (except metro_lines), and passageways */
INSERT INTO deleted_ft
SELECT ft_id
FROM ft
WHERE ft_type_id IN
(
    2206, /* transport-thread-stop */
    2205, /* transport-bus-thread */
    2212, /* transport-tram-thread */
    2213, /* transport-waterway-thread */
    2214, /* transport-metro-thread */
    2201, /* transport-bus-route */
    2202, /* transport-trolleybus-route */
    2203, /* transport-tram-route */
    2204, /* transport-minibus-route */
    2208, /* transport-waterway-route */
    2209, /* transport-metro-transition */
    2210  /* transport-metro-passageway */
);

CREATE INDEX ON deleted_ft (ft_id);

DELETE FROM ft_isocode
USING deleted_ft
WHERE ft_isocode.ft_id = deleted_ft.ft_id;

DELETE FROM ft_nm
USING deleted_ft
WHERE ft_nm.ft_id = deleted_ft.ft_id;

DELETE FROM ft_source
USING deleted_ft
WHERE ft_source.ft_id = deleted_ft.ft_id;

DELETE FROM ft_attr
USING deleted_ft
WHERE ft_attr.ft_id = deleted_ft.ft_id;

DELETE FROM ft_ft
USING deleted_ft
WHERE ft_ft.master_ft_id = deleted_ft.ft_id
OR ft_ft.slave_ft_id = deleted_ft.ft_id;

DELETE FROM ft_rd_el
USING deleted_ft
WHERE ft_rd_el.ft_id = deleted_ft.ft_id;

DELETE FROM node_isocode
USING deleted_ft, ft_center
WHERE ft_center.ft_id = deleted_ft.ft_id
AND ft_center.node_id = node_isocode.node_id;

DELETE FROM node
USING deleted_ft, ft_center
WHERE ft_center.ft_id = deleted_ft.ft_id
AND ft_center.node_id = node.node_id;

DELETE FROM ft_center
USING deleted_ft
WHERE ft_center.ft_id = deleted_ft.ft_id;

DELETE FROM ft_edge
USING deleted_ft
WHERE ft_edge.ft_id = deleted_ft.ft_id;

DELETE FROM face_edge
USING deleted_ft, ft_face
WHERE ft_face.ft_id = deleted_ft.ft_id
AND ft_face.face_id = face_edge.face_id;

DELETE FROM face_isocode
USING deleted_ft, ft_face
WHERE ft_face.ft_id = deleted_ft.ft_id
AND ft_face.face_id = face_isocode.face_id;

DELETE FROM face
USING deleted_ft, ft_face
WHERE ft_face.ft_id = deleted_ft.ft_id
AND ft_face.face_id = face.face_id;

DELETE FROM ft_face
USING deleted_ft
WHERE ft_face.ft_id = deleted_ft.ft_id;

DELETE FROM ft
USING deleted_ft
WHERE ft.ft_id = deleted_ft.ft_id;
