#include "config.h"
#include "log.h"
#include "region.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/transformers/record_common.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/stringutils/include/split.h>

#include <boost/filesystem.hpp>

#include <cctype>

namespace maps::wiki::json2ymapsdf::config {

namespace fs = boost::filesystem;

namespace {

class XmlConfig {
public:
    XmlConfig() = default;
    void init(const Params& params, const ymapsdf::schema::Schema& ymapsdfSchema);
    std::string cfgFile() const { return cfgFile_; }

    std::vector<const xml3::Node*> nodes(const tds::schema::Category& category) const;
    std::vector<const xml3::Node*> nodes(const tds::schema::Relation& tRel) const;
    std::string mainTableName(const tds::schema::Category& category) const;

    const NodeList& fixIncompleteNodes() const { return fixIncompleteNodes_; }
    const NodeList& fixYmapsdfNodes() const { return fixYmapsdfNodes_; }
    const NodeList& regionNodes() const { return regionNodes_; }
    const std::string& experiment() const {return experiment_; }

private:
    NodeList getNodeList(const std::string& path) const;
    void addCategoryNode(const xml3::Node& node);
    void addRelationNode(
        const ymapsdf::schema::Schema& ymapsdfSchema,
        const xml3::Node& node);

private:
    using CategoryNodesMap = std::multimap<std::string, xml3::Node>;
    using RelationNodesMap = std::multimap<tds::schema::RelationDescriptor, xml3::Node>;
    using CategoryMainTableMap = std::map<std::string, std::string>;

private:
    CategoryNodesMap categoryNodes_;
    RelationNodesMap relationNodes_;
    CategoryMainTableMap categoryMainTableMap_;
    NodeList fixIncompleteNodes_;
    NodeList fixYmapsdfNodes_;
    NodeList regionNodes_;

    std::shared_ptr<xml3::Doc> cfgDoc_;
    std::string cfgFile_;
    std::string experiment_;
};

XmlConfig&
xmlConfig()
{
    static XmlConfig xmlConfig;
    return xmlConfig;
}

const std::string ATTRIBUTE_DIRECTION = "direction";
const std::string ATTRIBUTE_INVERTED_ROLES = "inverted-roles";

const std::string ELEMENT_RECORD = "record";
const std::string DISABLE_IF_EXPERIMENT = "disable-if-experiment";
const std::string ENABLE_IF_EXPERIMENT = "enable-if-experiment";

const std::string PATH_EDITOR_CFG = "/json2ymapsdf/settings/editor-cfg";
const std::string PATH_FIX_INCOMPLETE_RECORD = "/json2ymapsdf/fix-incomplete/record";
const std::string PATH_FIX_YMAPSDF = "/json2ymapsdf/fix-ymapsdf/sql";
const std::string PATH_LINK = "/json2ymapsdf/transform/link";
const std::string PATH_OBJECT = "/json2ymapsdf/transform/object";
const std::string PATH_REGION = "/json2ymapsdf/regions/region";
const std::string PATH_RELATIONS_MODE = "/json2ymapsdf/settings/relations-mode";
const std::string PATH_SCHEMA_SQL = "/json2ymapsdf/settings/schema-sql";
const std::string PATH_SCHEMA_PUB_SQL = "/json2ymapsdf/settings/schema-pub-sql";
const std::string PATH_SCHEMA_VIEW_SQL = "/json2ymapsdf/settings/schema-view-sql";
const std::string PATH_VALIDATION_SQL_DIR = "/json2ymapsdf/settings/validation-sql-dir";


void
XmlConfig::init(const Params& params, const ymapsdf::schema::Schema& ymapsdfSchema)
{
    experiment_ = params.experiment;

    cfgFile_ = params.transformCfg;
    cfgDoc_ = std::make_shared<xml3::Doc>(cfgFile_);

    auto objectNodes = cfgDoc_->nodes(PATH_OBJECT, true);
    for (size_t i = 0; i < objectNodes.size(); ++i) {
        addCategoryNode(objectNodes[i]);
    }

    auto linkNodes = cfgDoc_->nodes(PATH_LINK, true);
    for (size_t i = 0; i < linkNodes.size(); ++i) {
        addRelationNode(ymapsdfSchema, linkNodes[i]);
    }

    fixIncompleteNodes_ = getNodeList(PATH_FIX_INCOMPLETE_RECORD);
    fixYmapsdfNodes_ = getNodeList(PATH_FIX_YMAPSDF);
    regionNodes_ = getNodeList(PATH_REGION);
}


NodeList
XmlConfig::getNodeList(const std::string& path) const
{
    auto nodes = cfgDoc_->nodes(path, true);
    NodeList result;
    for (size_t nodesIx = 0; nodesIx < nodes.size(); ++nodesIx) {
        result.push_back(nodes[nodesIx]);
    }
    return result;
}

bool
disabled(const std::string& experiment, const xml3::Node& node)
{
    const auto experimentToEnable = getValue(node, ENABLE_IF_EXPERIMENT);
    const auto experimentToDisable = getValue(node, DISABLE_IF_EXPERIMENT);

    return !experimentToEnable.empty() && experimentToEnable != experiment
        || !experimentToDisable.empty() && experimentToDisable == experiment;
}

void
XmlConfig::addRelationNode(
    const ymapsdf::schema::Schema& ymapsdfSchema,
    const xml3::Node& node)
{
    if (disabled(experiment_, node)) {
        return;
    }

    for (const auto& relation: getRelations(ymapsdfSchema, node)) {
        auto recordNodes = node.nodes(ELEMENT_RECORD, true);
        for (size_t n = 0; n < recordNodes.size(); ++n) {
            if (disabled(experiment_, recordNodes[n])) {
                continue;
            }
            relationNodes_.insert({relation, recordNodes[n]});
        }
        if (recordNodes.size() == 0) {
            relationNodes_.insert({relation, node});
        }
    }
}

void
XmlConfig::addCategoryNode(const xml3::Node& node)
{
    if (disabled(experiment_, node)) {
        return;
    }

    auto mainTblName = config::mainTableName(node);

    for (const auto& catName: getCategories(node)) {
        auto recordNodes = node.nodes(ELEMENT_RECORD, true);
        for (size_t i = 0; i < recordNodes.size(); ++i) {
            if (disabled(experiment_, recordNodes[i])) {
                continue;
            }
            categoryNodes_.insert({catName, recordNodes[i]});
        }
        if (recordNodes.size() == 0) {
            categoryNodes_.insert({catName, node});
        }

        if (!mainTblName.empty()) {
            categoryMainTableMap_.insert({catName, mainTblName});
        }
    }
}

std::vector<const xml3::Node*>
XmlConfig::nodes(const tds::schema::Category& category) const
{
    std::vector<const xml3::Node*> nodes;
    auto range = categoryNodes_.equal_range(category.name());
    for (auto it = range.first; it != range.second; ++it) {
        nodes.push_back(&it->second);
    }
    return nodes;
}

std::vector<const xml3::Node*>
XmlConfig::nodes(const tds::schema::Relation& tRel) const
{
    //collect entirely specified relations
    std::vector<const xml3::Node*> nodes;
    auto range = relationNodes_.equal_range({tRel.master().name(), tRel.role()});
    for (auto it = range.first; it != range.second; ++it) {
        nodes.push_back(&it->second);
    }
    //collect relations with role only
    range = relationNodes_.equal_range({std::string(), tRel.role()});
    for (auto it = range.first; it != range.second; ++it) {
        nodes.push_back(&it->second);
    }
    return nodes;
}

std::string
XmlConfig::mainTableName(const tds::schema::Category& category) const
{
    auto it = categoryMainTableMap_.find(category.name());
    return it == categoryMainTableMap_.end()
        ? std::string()
        : it->second;
}

} // namespace

void
init(const Params& params, const ymapsdf::schema::Schema& ymapsdfSchema)
{
    return xmlConfig().init(params, ymapsdfSchema);
}

bool
disabled(const xml3::Node& node)
{
    return disabled(xmlConfig().experiment(), node);
}

std::string
cfgDir()
{
    fs::path cfgPath(xmlConfig().cfgFile());
    return cfgPath.parent_path().string();
}

std::vector<const xml3::Node*>
nodes(const tds::schema::Category& category)
{
    return xmlConfig().nodes(category);
}

std::vector<const xml3::Node*>
nodes(const tds::schema::Relation& tdsRelation)
{
    return xmlConfig().nodes(tdsRelation);
}

std::vector<const xml3::Node*>
nodes(
    const tds::schema::Relation& tdsRelation,
    const ymapsdf::schema::Table& defaultTable)
{
    std::vector<const xml3::Node*> nodes;
    for (const auto* node:  xmlConfig().nodes(tdsRelation)) {
        if (!masterName(*node).empty()
            || defaultTable.name() == tableName(*node)
            || tableName(*node).empty() )
        {
            nodes.push_back(node);
        }
    }
    return nodes;
}

std::string
mainTableName(const tds::schema::Category& category)
{
    return xmlConfig().mainTableName(category);
}

const NodeList&
fixIncompleteNodes()
{
    return xmlConfig().fixIncompleteNodes();
}

const NodeList&
fixYmapsdfNodes()
{
    return xmlConfig().fixYmapsdfNodes();
}

const NodeList&
regionNodes()
{
    return xmlConfig().regionNodes();
}

bool
toSkip(const tds::schema::Category& tdsCategory)
{
    for (auto cfgNode: config::nodes(tdsCategory)) {
        if (config::skipAlgorithm(*cfgNode)) {
            return true;
        }
    }

    return false;
}

bool
toSkip(const tds::schema::Relation& tdsRelation)
{
    for (auto cfgNode: config::nodes(tdsRelation)) {
        if (config::skipAlgorithm(*cfgNode)) {
            return true;
        }
    }

    return toSkip(tdsRelation.master()) || toSkip(tdsRelation.slave());
}

std::string
getPathEditorCfg(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    return doc.node(PATH_EDITOR_CFG).value<std::string>();
}

std::string
getPathSchemaSql(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    return doc.node(PATH_SCHEMA_SQL).value<std::string>();
}

std::string
getPathSchemaPubSql(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    return doc.node(PATH_SCHEMA_PUB_SQL, true).isNull()
        ? getPathSchemaSql(cfgFile)
        : doc.node(PATH_SCHEMA_PUB_SQL).value<std::string>();
}

std::string
getPathSchemaViewSql(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    return doc.node(PATH_SCHEMA_VIEW_SQL).value<std::string>();
}

std::string
getPathValidationSqlDir(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    return doc.node(PATH_VALIDATION_SQL_DIR, true).value<std::string>(std::string());
}

tds::schema::RelationsMode
relationsMode(const std::string& cfgFile)
{
    xml3::Doc doc{cfgFile};
    auto relModeNode = doc.node(PATH_RELATIONS_MODE);

    auto direction = relModeNode.attr<tds::schema::RelationsDirection>(
        ATTRIBUTE_DIRECTION,
        tds::schema::RelationsDirection::Unknown);

    auto invertedRoles = getValue(relModeNode, ATTRIBUTE_INVERTED_ROLES);

    return tds::schema::RelationsMode(direction, invertedRoles);
}

const std::string& experiment()
{
    return xmlConfig().experiment();
}

} // namespace maps::wiki::json2ymapsdf::config
