#include "config.h"
#include "log.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/transformers/field_common.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/ymapsdf/schema/pqxx.h>

#include <maps/libs/common/include/file_utils.h>

namespace maps::wiki::json2ymapsdf::config {

namespace {

const std::string ATTRIBUTE_ALGORITHM = "algorithm";
const std::string ELEMENT_FIELD = "field";
const std::string ATTRIBUTE_FIELD = "column";
const std::string ATTRIBUTE_TYPE = "type";

const std::string ATTRIBUTE_SOURCE = "source";
const std::string ATTRIBUTE_VALUE = "value";

const std::string ELEMENT_WHEN = "when";
const std::string ATTRIBUTE_CATEGORY = "category";
const std::string ATTRIBUTE_ROLE = "role";
const std::string ATTRIBUTE_MASTER = "master";

const std::string ATTRIBUTE_TABLE = "table";
const std::string ATTRIBUTE_MAIN_TABLE = "main-table";
const std::string ELEMENT_IF = "if";

const std::string VALUE_MASTER_OBJECT_ID = "MASTER_OBJECT_ID";
const std::string VALUE_SLAVE_OBJECT_ID = "SLAVE_OBJECT_ID";

const std::string ATTRIBUTE_DIRECTION = "direction";
const std::string ATTRIBUTE_INVERTED_ROLES = "inverted-roles";

const std::string MULTI_CATEGORY_ERROR =
    "More then one of category/role specifiers "\
    "(attrs 'role', 'category', elements 'role', 'category')";

const std::string STR_ALLOW_EMPTY_SETTING = "allow-empty";
const std::string STR_TRUE = "true";
const std::string STR_ONE = "1";

void
addCategoryConstAttrs(const xml3::Node& whenNode)
{
    auto catName = getValue(whenNode, ATTRIBUTE_CATEGORY);
    auto& category = tds::schema::category(catName);

    for (const auto& attr: config::getAttrs(whenNode)) {
        if (attr != ATTRIBUTE_CATEGORY) {
            category.addConstAttr(attr, config::getValue(whenNode, attr));
        }
    }
}

void addRelationConstAttrs(const xml3::Node& whenNode)
{
    auto role = getValue(whenNode, ATTRIBUTE_ROLE);
    auto master = getValue(whenNode, ATTRIBUTE_MASTER);

    for (const auto& attr: config::getAttrs(whenNode)) {
        if (attr == ATTRIBUTE_MASTER || attr == ATTRIBUTE_ROLE) {
            continue;
        }
        for (const auto& relation : tds::schema::getRelations(master, role)) {
            relation->addConstAttr(attr, config::getValue(whenNode, attr));
        }
    }
}

} // namespace

std::vector<std::string>
getCategories(const xml3::Node& cfgNode)
{
    auto catName = getValue(cfgNode, ATTRIBUTE_CATEGORY);
    auto whenNodes = cfgNode.nodes(ELEMENT_WHEN, true);

    if (!catName.empty()) {
        REQUIRE(whenNodes.size() == 0, MULTI_CATEGORY_ERROR << " in " << cfgNode.path());
        tds::schema::category(catName); // validate
        return {catName};
    }

    std::vector<std::string> categories;
    for (size_t i = 0; i < whenNodes.size(); ++i) {
        const auto& whenNode = whenNodes[i];
        catName = getValue(whenNode, ATTRIBUTE_CATEGORY);
        if (tds::schema::categories().count(catName) == 0) {
            WARN() << "Skip unknown category '" << catName << "'";
            continue;
        }
        tds::schema::category(catName); // validate
        categories.emplace_back(catName);
        addCategoryConstAttrs(whenNode);
    }
    return categories;
}

std::vector<tds::schema::RelationDescriptor>
getRelations(const ymapsdf::schema::Schema& ymapsdfSchema, const xml3::Node& cfgNode)
{
    auto role = getValue(cfgNode, ATTRIBUTE_ROLE);
    auto master = getValue(cfgNode, ATTRIBUTE_MASTER);
    auto whenNodes = cfgNode.nodes(ELEMENT_WHEN, true);

    if (!role.empty()) {
        REQUIRE(whenNodes.size() == 0, MULTI_CATEGORY_ERROR << " in " << cfgNode.path());
        if (!skipAlgorithm(cfgNode)) {
            tds::schema::checkRelation(master, role);
        } else if (tds::schema::getRelations(master, role).empty()) {
            tds::schema::addRelation(ymapsdfSchema, master, role, master);
        }
        return {{master, role}};
    }

    std::vector<tds::schema::RelationDescriptor> relations;
    for (size_t i = 0; i < whenNodes.size(); ++i) {
        const auto& whenNode = whenNodes[i];
        auto role = getValue(whenNode, ATTRIBUTE_ROLE);
        auto master = getValue(whenNode, ATTRIBUTE_MASTER);
        tds::schema::checkRelation(master, role);
        relations.emplace_back(master, role);
        addRelationConstAttrs(whenNode);
    }
    return relations;
}

std::vector<std::string>
getAttrs(const xml3::Node& cfgNode)
{
    std::vector<std::string> attrs;
    for (const auto& attr: cfgNode.attrNames()) {
        if (attr != ATTRIBUTE_SOURCE && attr != ATTRIBUTE_VALUE && attr != ATTRIBUTE_ALGORITHM)
        {
            attrs.push_back(attr);
        }
    }
    return attrs;
}

std::string
getValue(const xml3::Node& node, const std::string& property)
{
    return node.attr<std::string>(property, std::string());
}

std::string
algorithmName(const xml3::Node& node)
{
    return getValue(node, ATTRIBUTE_ALGORITHM);
}

bool
skipAlgorithm(const xml3::Node& node)
{
    return algorithmName(node) == transformers::SKIP_RECORD;
}

std::string
masterName(const xml3::Node& recordNode)
{
    return getValue(recordNode, ATTRIBUTE_MASTER);
}

std::string
tableName(const xml3::Node& recordNode)
{
    std::string tableName = getValue(recordNode, ATTRIBUTE_TABLE);
    if (tableName.empty()) {
        return mainTableName(recordNode);
    }
    return tableName;
}

std::string
mainTableName(const xml3::Node& recordNode)
{
    return getValue(recordNode, ATTRIBUTE_MAIN_TABLE);
}

const ymapsdf::schema::Table*
table(const ymapsdf::schema::Schema& ymapsdfSchema, const xml3::Node& recordNode)
{
    auto tblName = tableName(recordNode);
    if (tblName.empty()) {
        return nullptr;
    }
    return &ymapsdfSchema.table(tblName);
}

std::string
fieldName(const xml3::Node& fieldNode)
{
    return getValue(fieldNode, ATTRIBUTE_FIELD);
}

std::string
attrName(const xml3::Node& fieldNode)
{
    return getValue(fieldNode, ATTRIBUTE_SOURCE);
}

ymapsdf::schema::Type
fieldType(const xml3::Node& fieldNode)
{
    return boost::lexical_cast<ymapsdf::schema::Type>(getValue(fieldNode, ATTRIBUTE_TYPE));
}

std::string
value(const xml3::Node& fieldNode)
{
    return getValue(fieldNode, ATTRIBUTE_VALUE);
}

std::string
recordTransformerName(const xml3::Node& recordNode)
{
    if (!algorithmName(recordNode).empty()) {
        return algorithmName(recordNode);
    }
    return transformers::COMMON_RECORD;
}

std::string
fieldTransformerName(const xml3::Node& fieldNode)
{
    if (!algorithmName(fieldNode).empty()) {
        return algorithmName(fieldNode);
    }
    return transformers::COMMON_FIELD;
}

std::string
sql(const xml3::Node& node)
{
    auto sqlValue = getValue(node, ATTRIBUTE_VALUE);
    if (!sqlValue.empty()) {
        return sqlValue;
    }

    const auto sqlFile = getValue(node, ATTRIBUTE_SOURCE);
    REQUIRE(!sqlFile.empty(), "No source and value attributes for fix-ymapsdf/sql element");
    return maps::common::readFileToString(cfgDir() + "/" + sqlFile);
}


xml3::Nodes
fieldNodes(const xml3::Node& recordNode)
{
    return recordNode.nodes(ELEMENT_FIELD, true);
}

xml3::Nodes
ifNodes(const xml3::Node& recordNode)
{
    return recordNode.nodes(ELEMENT_IF, true);
}

bool
corresponds(const ymapsdf::schema::Relation& yRel, const xml3::Node* node)
{
    if (tableName(*node) != yRel.table().name()) {
        return false;
    }

    bool matched = false;
    xml3::Nodes attrNodes = node->nodes(ELEMENT_FIELD, true);
    for (size_t i = 0; i < attrNodes.size(); ++i) {
        auto attr = attrName(attrNodes[i]);
        auto field = fieldName(attrNodes[i]);
        if (field == ymapsdf::PRIMARY_KEY) {
            continue;
        }
        if (attr == VALUE_MASTER_OBJECT_ID) {
            if (field !=  yRel.master().name()) {
                return false;
            }
            matched = true;
        } else if (attr == VALUE_SLAVE_OBJECT_ID) {
            if (field !=  yRel.slave().name()) {
                return false;
            }
            matched = true;
        }
    }
    return matched;
}

bool
allowEmptySetting(const xml3::Node* node)
{
    const auto value = getValue(*node, STR_ALLOW_EMPTY_SETTING);
    return value == STR_TRUE || value == STR_ONE;
}

} // namespace maps::wiki::json2ymapsdf::config
