#include "json_helper.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/work/transform/files_queue.h>

#include <yandex/maps/wiki/common/geom.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/variant.h>

namespace maps::wiki::json2ymapsdf::tds {

namespace {

const std::string JSON_FIELD_OBJECTS = "objects";
const std::string JSON_FIELD_TYPE = "type";
const std::string JSON_FIELD_ATTRIBUTES = "attributes";
const std::string JSON_FIELD_RELATIONS = "relations";
const std::string JSON_NEXT_FREE_OBJECT_ID = "next_free_object_id";
const std::string JSON_RELATIONS_EXPORT_MODE = "relations_export_mode";
const std::string JSON_INVERT_DIRECTION_FOR_ROLES = "invert_direction_for_roles";
const std::string JSON_MASTER_TO_SLAVE = "Master ==> Slave";
const std::string JSON_SLAVE_TO_MASTER = "Master <== Slave";

const std::string CATEGORY_PREFIX = "cat:";
const char CHAR_COLON = ':';

} // namespace

std::string
geoJson2Wkb(const json::Value& geoJson) {
    try {
        auto geometry = geolib3::readGeojson<geolib3::SimpleGeometryVariant>(geoJson);
        return geolib3::WKB::toString(geometry);
    } catch (const std::exception& ex) {
        throw TdsDataError() << "Unable to process geometry: " << ex.what();;
    }
}

std::map<std::string, std::string>
readAttributes(const json::Value& object)
{
    std::map<std::string, std::string> attributes;
    if (object.hasField(JSON_FIELD_ATTRIBUTES)) {
        const auto& attrObject = object[JSON_FIELD_ATTRIBUTES];
        auto keys = attrObject.fields();
        for (const auto& key : keys) {
            attributes[attributeName(key)] = attrObject[key].toString();
        }
    }
    return attributes;
}

schema::Categories
readCategories(const json::Value& object)
{
    schema::Categories categories;
    auto attrNames = object[JSON_FIELD_ATTRIBUTES].fields();
    for (const auto& attrName : attrNames) {
        if (attrName.starts_with(CATEGORY_PREFIX)) {
            auto catName = attrName.substr(CATEGORY_PREFIX.length());
            categories.insert(&schema::category(catName));
        }
    }
    REQUIRE(categories.size(), "Object without category");
    return categories;
}

std::string
categoryName(const std::string& attr)
{
    if (attr.starts_with(CATEGORY_PREFIX)) {
        return attr.substr(CATEGORY_PREFIX.length());
    }
    return attr.substr(0, attr.find(CHAR_COLON));
}

std::string
attributeName(const std::string& attr)
{
    std::string category = categoryName(attr);
    if (category.size() < attr.size()
        && attr.starts_with(category))
    {
        return attr.substr(category.size() + 1);
    }
    return std::string();
}

const json::Value&
getObjects(const json::Value& json)
{
    ASSERT(json.hasField(JSON_FIELD_OBJECTS));
    return json[JSON_FIELD_OBJECTS];
}

const json::Value&
getLinks(const json::Value& jsonObject)
{
    static json::Value emptyLinks{std::vector<json::Value>()};
    if(!jsonObject.hasField(JSON_FIELD_RELATIONS)) {
        return emptyLinks;
    }
    ASSERT(jsonObject[JSON_FIELD_RELATIONS].isArray());
    return jsonObject[JSON_FIELD_RELATIONS];
}

DBID
nextFreeObjectId(const json::Value& json)
{
    const auto& attributes = json[JSON_FIELD_ATTRIBUTES];
    if (attributes.hasField(JSON_NEXT_FREE_OBJECT_ID)) {
        return std::stoul(attributes[JSON_NEXT_FREE_OBJECT_ID].toString());
    }
    DBID maxId = 0;
    const auto keys = getObjects(json).fields();
    for (const auto& key : keys) {
        maxId = std::max(maxId, std::stol(key));
    }
    return maxId + 1;
}

schema::RelationsMode
relationsMode(const json::Value& json)
{
    std::set<std::string> roles;
    schema::RelationsDirection direction = schema::RelationsDirection::Unknown;

    const auto& attributes = json[JSON_FIELD_ATTRIBUTES];

    if (attributes.hasField(JSON_RELATIONS_EXPORT_MODE)) {
        auto directionName = attributes[JSON_RELATIONS_EXPORT_MODE].toString();
        direction = boost::lexical_cast<schema::RelationsDirection>(directionName);
    }

    std::string invertedRoles;
    if (attributes.hasField(JSON_INVERT_DIRECTION_FOR_ROLES)) {
        invertedRoles = attributes[JSON_INVERT_DIRECTION_FOR_ROLES].toString();
    }

    return schema::RelationsMode(direction, invertedRoles);
}

json::Value
readJson(const std::string& jsonFile)
{
    if (jsonFile == STDIN_FILENAME) {
        return json::Value::fromStream(std::cin);
    }
    return json::Value::fromFile(jsonFile);
}

} // namespace maps::wiki::json2ymapsdf::tds
