#include "fix_incomplete.h"
#include "configuration.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/config.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/work/transform/id_manager.h>

namespace maps::wiki::json2ymapsdf::transformers {

ymapsdf::Records
fix(const ymapsdf::schema::Schema& ymapsdfSchema, ymapsdf::Record&& record)
{
    try {
        return fixer(record.table()).fix(ymapsdfSchema, std::move(record));
    } catch (std::exception &e) {
        DATA_ERROR() << "Fixing error: " << record.table().insertClause() << " " << record;
        throw TdsDataError() << "Fixing error: " << e.what();
    }
}


namespace {

const std::string STR_FT = "ft";
const std::string STR_FT_EDGE = "ft_edge";
const std::string STR_EDGE_ID = "edge_id";
const std::string STR_FT_ID = "ft_id";
const std::string STR_FT_TYPE_ID = "ft_type_id";

const std::string STR_COND_RD_SEQ = "cond_rd_seq";
const std::string STR_COND_SEQ_ID = "cond_seq_id";
const std::string STR_COND = "cond";
const std::string STR_COND_ID = "cond_id";
const std::string STR_COND_TYPE = "cond_type";

using makeFixerFunc = std::shared_ptr<RecordFixer> ();

template<class Fixer>
std::shared_ptr<RecordFixer>
makeFixer()
{
    return std::make_shared<Fixer>();
}

} // namespace

std::shared_ptr<RecordFixer>
RecordFixer::create(const xml3::Node* node)
{
    const static std::map<std::string, makeFixerFunc&> fixerBuilders =
    {
        {DROP_FIXER, makeFixer<DropFixer>},
        {REFUSE_FIXER, makeFixer<RefuseFixer>},
        {FAKE_FT_FIXER, makeFixer<FakeFtFixer>},
        {TRAFFIC_LIGHT, makeFixer<TrafficLightFixer>},
    };

    const auto it = fixerBuilders.find(config::algorithmName(*node));
    REQUIRE(it != fixerBuilders.end(),
        "Unknown fixer algorithm: '" << config::algorithmName(*node) << "'");
    return (it->second)();
}

ymapsdf::Records
DropFixer::fix(const ymapsdf::schema::Schema&, ymapsdf::Record&& record) const
{
    ymapsdf::Records records;
    if (record.valid()) {
        records.push_back(std::move(record));
    }
    return records;
}

namespace {

std::string
invalidColumns(const ymapsdf::Record& record)
{
    std::string str;
    for (size_t i = 0; i < record.size(); ++i) {
        if (!record[i].valid()) {
            str += str.empty() ? "" : " ";
            str += record[i].column().name();
        }
    }
    return str;
}

} // namespace

ymapsdf::Records
RefuseFixer::fix(const ymapsdf::schema::Schema&, ymapsdf::Record&& record) const
{
    ymapsdf::Records records;

    DATA_REQUIRE(record.valid(),
        "Invalid record " << record.table().name() << ". Columns ["
        << invalidColumns(record) << "] must not be NULL " << record);

    records.push_back(std::move(record));
    return records;
}

ymapsdf::Records
FakeFtFixer::fix(const ymapsdf::schema::Schema& ymapsdfSchema, ymapsdf::Record&& ftEdgeRecord) const
{
    REQUIRE(&ftEdgeRecord.table() == &ymapsdfSchema.table(STR_FT_EDGE),
        "Wrong fixing record:" << ftEdgeRecord.table().name()
        << ". Only ft_edge is supported");
    ymapsdf::Records records;
    ftEdgeRecord[STR_FT_ID] = *ftEdgeRecord[STR_EDGE_ID];
    ymapsdf::Record ftRecord(ymapsdfSchema, STR_FT);
    ftRecord[STR_FT_ID] = *ftEdgeRecord[STR_EDGE_ID];
    ftRecord[STR_FT_TYPE_ID] = *ftEdgeRecord[STR_FT_TYPE_ID];
    records.push_back(std::move(ftEdgeRecord));
    records.push_back(std::move(ftRecord));
    return records;
}

ymapsdf::Records
TrafficLightFixer::fix(const ymapsdf::schema::Schema& ymapsdfSchema, ymapsdf::Record&& condRdSeqRecord) const
{
    REQUIRE(&condRdSeqRecord.table() == &ymapsdfSchema.table(STR_COND_RD_SEQ),
        "Wrong fixing record:" << condRdSeqRecord.table().name()
        << ". Only cond_rd_seq is supported");
    const auto condId = idManager().uniqueDBID();
    ymapsdf::Records records;
    condRdSeqRecord[STR_COND_SEQ_ID] = condId;
    ymapsdf::Record condRecord(ymapsdfSchema, STR_COND);
    condRecord[STR_COND_ID] = condId;
    condRecord[STR_COND_SEQ_ID] = condId;
    condRecord[STR_COND_TYPE] = *condRdSeqRecord[STR_COND_TYPE];
    records.push_back(std::move(condRdSeqRecord));
    records.push_back(std::move(condRecord));
    return records;
}

} // namespace maps::wiki::json2ymapsdf::transformers
