#include "record_entrance_level_flat_range.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>

#include <maps/libs/json/include/builder.h>
#include <maps/wikimap/mapspro/libs/flat_range/include/distribution.h>
#include <maps/wikimap/mapspro/libs/flat_range/include/range.h>

#include <algorithm>
#include <cctype>

namespace maps::wiki::json2ymapsdf::transformers {

namespace {

const std::string ENTRANCE_LEVEL_FLAT_RANGE_TABLE = "entrance_level_flat_range";
const std::string STR_FLAT_FIRST = "flat_first";
const std::string STR_FLAT_LAST = "flat_last";
const std::string STR_LEVEL_UNIVERSAL_ID = "level_universal_id";
const std::string STR_IS_EXACT = "is_exact";
const std::string STR_FALSE = "false";
const std::string STR_TRUE = "true";
const std::string STR_LEVEL_FLAT_RANGE_ID = "level_flat_range_id";

std::string jsonMeta(const std::string& universalId, DBID rangeId)
{
    maps::json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["level_universal_id"] = universalId;
        builder["flat_range_id"] = rangeId;
    };
    return builder.str();
}

ymapsdf::Record
createLevelFlatRangeRecord(
    const ymapsdf::schema::Schema& schema,
    const std::string& objectId,
    const std::string& flatFirst,
    const std::string& flatLast,
    const std::string& universalId)
{
    auto flatsRecord = ymapsdf::Record::defaultRecord(schema, ENTRANCE_LEVEL_FLAT_RANGE_TABLE);
    flatsRecord[STR_LEVEL_FLAT_RANGE_ID] = objectId;
    flatsRecord[STR_FLAT_FIRST] = flatFirst;
    flatsRecord[STR_FLAT_LAST] = flatLast;
    flatsRecord[STR_LEVEL_UNIVERSAL_ID] = universalId;
    flatsRecord[STR_IS_EXACT] = STR_TRUE;
    return flatsRecord;
}

void
createLevelFlatRangesRecords(
    const ymapsdf::schema::Schema& schema,
    const std::string& objectId,
    const flat_range::Ranges& flatRanges,
    const std::string& universalId,
    ymapsdf::Records& records)
{
    for (const auto& flatRange : flatRanges) {
        if (flatRange.named()) {
            for (size_t flatNum = 0; flatNum < flatRange.size(); ++flatNum) {
                records.push_back(
                    createLevelFlatRangeRecord(
                        schema,
                        objectId,
                        flatRange.value(flatNum),
                        flatRange.value(flatNum),
                        jsonMeta(universalId, idManager().uniqueDBID())));
            }
        } else {
            records.push_back(
                createLevelFlatRangeRecord(
                    schema,
                    objectId,
                    flatRange.min(),
                    flatRange.max(),
                    jsonMeta(universalId, idManager().uniqueDBID())));
        }
    }
}

} // namespace

ymapsdf::Records
EntranceLevelFlatRangeTransformer::transform(const tds::Item& object) const
{
    ymapsdf::Records records;
    const auto objectId = object[tds::ATTRIBUTE_OBJECT_ID];

    try {
        createLevelFlatRangesRecords(
            ymapsdfSchema_,
            objectId,
            flat_range::parse(object["flats"]),
            {},
            records);

        auto levelsWithFlats = flat_range::distributeFlatsByLevels(
            {{object["flats"], object["levels"]}}).levelsWithFlats;
        if (levelsWithFlats.empty()) {
            return records;
        }

        for (const auto& levelWithFlats : levelsWithFlats) {
            createLevelFlatRangesRecords(
                ymapsdfSchema_,
                objectId,
                levelWithFlats.flatRanges,
                levelWithFlats.levelName,
                records);
        }
    } catch (const flat_range::ParseException& ex) {
        WARN() << "FLAT_RANGE Failed to parse flat_range: " << objectId << " " << ex;
    } catch (const TdsDataError& ex) {
        WARN() << "FLAT_RANGE Failed to process flat_range: " << objectId << " " << ex;
    }
    return records;
}

ymapsdf::schema::TableSet
EntranceLevelFlatRangeTransformer::tables() const
{
    return {&ymapsdfSchema_.table(ENTRANCE_LEVEL_FLAT_RANGE_TABLE)};
}

EntranceLevelFlatRangeTransformer::EntranceLevelFlatRangeTransformer(
    const ymapsdf::schema::Schema& ymapsdfSchema,
    const ymapsdf::schema::Table*,
    const tds::schema::Category*,
    const xml3::Node*)
    : RecordTransformer{ymapsdfSchema}
{ }

} // namespace maps::wiki::json2ymapsdf::transformers
