#include "record_line.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/log.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/work/transform/id_manager.h>
#include <yandex/maps/wiki/common/geom.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/serialization.h>

namespace maps::wiki::json2ymapsdf::transformers {

namespace {

const std::string STR_EDGE = "edge";
const std::string STR_NODE = "node";
const std::string STR_FT_ID = "ft_id";
const std::string STR_FT_EDGE = "ft_edge";
const std::string STR_EDGE_ID = "edge_id";
const std::string STR_SHAPE = "shape";
const std::string STR_F_NODE_ID = "f_node_id";
const std::string STR_T_NODE_ID = "t_node_id";
const std::string STR_F_ZLEV = "f_zlev";
const std::string STR_T_ZLEV = "t_zlev";
const std::string STR_NODE_ID = "node_id";

} // namespace

namespace {
const ymapsdf::schema::Column& objIdColumn(
    const tds::schema::Category& category,
    const ymapsdf::schema::Table& table)
{
    REQUIRE(category.tablePtr() != nullptr, "Category " << category.name()
        << " has no associated table, unable to configure LINE transformer");
    for (size_t i = 0; i < table.size(); ++i) {
        const ymapsdf::schema::Column& column = table[i];
        if (category.tablePtr() == column.foreignPtr()) {
            return column;
        }
    }
    throw LogicError() << "Table " << table.name() << " has no column references to '"
        << category.name() << "' category";
}
} // namespace

LineTransformer::LineTransformer(
    const ymapsdf::schema::Schema& ymapsdfSchema,
    const ymapsdf::schema::Table* table,
    const tds::schema::Category* category,
    const xml3::Node*)
        : RecordTransformer{ymapsdfSchema}
        , objEdgeTable_(*table)
        , objIdColumn_(objIdColumn(*category, objEdgeTable_))
{ }

ymapsdf::Records
LineTransformer::transform(const tds::Item& object) const
{
    const auto objId = std::stoul(object[tds::ATTRIBUTE_OBJECT_ID]);
    const auto& shape = object[tds::ATTRIBUTE_SHAPE];
    if (shape.empty()) {
        WARN() << "Line object " << objId << " without geometry";
        return {};
    }
    try {
        ymapsdf::Records records;
        processLine(objId, shape, records);
        return records;
    } catch (const std::exception& ex) {
        throw TdsDataError() << "Unable to process line geometry: " << ex.what();
    }
}

ymapsdf::schema::TableSet
LineTransformer::tables() const
{
    return {
        &ymapsdfSchema_.table(STR_FT_EDGE),
        &ymapsdfSchema_.table(STR_EDGE),
        &ymapsdfSchema_.table(STR_NODE)
    };
}

void
LineTransformer::processLine(
    const DBID objId,
    const std::string& shape,
    ymapsdf::Records& records) const
{
    const auto edgeId = idManager().uniqueDBID();
    auto ftEdgeRecord = ymapsdf::Record::defaultRecord(ymapsdfSchema_, STR_FT_EDGE);
    ftEdgeRecord[STR_FT_ID] = objId;
    ftEdgeRecord[STR_EDGE_ID] = edgeId;
    records.push_back(std::move(ftEdgeRecord));

    const auto fNodeId = idManager().uniqueDBID();
    const auto tNodeId = idManager().uniqueDBID();

    auto edgeRecord = ymapsdf::Record::defaultRecord(ymapsdfSchema_, STR_EDGE);
    edgeRecord[STR_EDGE_ID] = edgeId;
    edgeRecord[STR_SHAPE] = shape;
    edgeRecord[STR_F_NODE_ID] = fNodeId;
    edgeRecord[STR_T_NODE_ID] = tNodeId;
    records.push_back(std::move(edgeRecord));

    const auto edgePolyline = geolib3::WKB::read<geolib3::Polyline2>(shape);

    auto fNodeWkb = geolib3::WKB::toString(edgePolyline.pointAt(0));
    auto tNodeWkb = geolib3::WKB::toString(edgePolyline.pointAt(edgePolyline.pointsNumber()-1));

    auto fNodeRecord = ymapsdf::Record::defaultRecord(ymapsdfSchema_, STR_NODE);
    fNodeRecord[STR_NODE_ID] = fNodeId;
    fNodeRecord[STR_SHAPE] = fNodeWkb;
    records.push_back(std::move(fNodeRecord));

    auto tNodeRecord = ymapsdf::Record::defaultRecord(ymapsdfSchema_, STR_NODE);
    tNodeRecord[STR_NODE_ID] = tNodeId;
    tNodeRecord[STR_SHAPE] = tNodeWkb;
    records.push_back(std::move(tNodeRecord));
}

} // namespace maps::wiki::json2ymapsdf::transformers
