#include "geometry_processor.h"
#include "helpers.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/helpers.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/coverage5/coverage.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/serialization.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>

#include <boost/none.hpp>

#include <cstdint>
#include <deque>
#include <pqxx/pqxx>
#include <unordered_set>
#include <utility>
#include <vector>


namespace maps::wiki::json2ymapsdf::isocode {

namespace {

using ID = int64_t;
using Isocodes = std::set<std::string>;

const std::string UNKNOWN_ISOCODE_001 = "001";
const size_t IDS_BATCH_SIZE = 10000;

} // namespace


class GeometryProcessor::Impl
{
public:
    Impl(
        pgpool3::Pool& pool,
        const std::string& coverageFile,
        std::string schemaName,
        std::shared_ptr<std::atomic<bool>> fail)
            : pool_(pool)
            , coverage_(coverageFile)
            , layer_(coverage_[coverage_.layersNames().front()])
            , schemaName_(std::move(schemaName))
            , fail_(std::move(fail))
    {
        checkRegions();
    }

    void processRdEl(size_t threads) const
    {
        createIsocodes<geolib3::Polyline2>("rd_el", "rd_el_id", threads);
    }

    void processRdJc(size_t threads) const
    {
        createIsocodes<geolib3::Point2>("rd_jc", "rd_jc_id", threads);
    }

    void processEdge(size_t threads) const
    {
        createIsocodes<geolib3::Polyline2>("edge", "edge_id", threads);
    }

    void processNode(size_t threads) const
    {
        createIsocodes<geolib3::Point2>("node", "node_id", threads);
    }

private:
    template <typename GeomType>
    void createIsocodes(
        const std::string& tableName,
        const std::string& idName,
        size_t threads) const
    {
        if (*fail_) {
            return;
        }

        ASSERT(threads > 1);
        INFO() << tableName << " start, threads: " << threads;

        ThreadPool threadPool(threads - 1);

        createIsocodes<GeomType>(tableName, idName, threadPool);
        INFO() << tableName << " waiting";
        threadPool.shutdown();

        INFO() << tableName << " processed";
    }

    std::unordered_set<ID> loadExistedIds(
        const std::string& tableName,
        const std::string& idName) const
    {
        std::unordered_set<ID> existedIds;
        auto query =
            "SELECT " + idName +
            " FROM " + schemaName_ + "." + tableName + "_isocode";

        INFO() << tableName << " loading existing data";
        auto rows = readWithRetries(pool_, query);
        for (const auto& row : rows) {
            existedIds.insert(row[0].as<ID>());
        }
        INFO() << tableName << " existing isocodes, size: "  << existedIds.size();
        return existedIds;
    }

    template <typename GeomType>
    void createIsocodes(
        const std::string& tableName,
        const std::string& idName,
        ThreadPool& threadPool) const
    {
        auto existedIds = loadExistedIds(tableName, idName);

        size_t count = 0;
        ID lastId = 0;
        for (;;) {
            std::vector<ID> ids;
            std::ostringstream query;
            query << "SELECT " << idName << " FROM " << schemaName_ << "." << tableName;
            query << " WHERE " << idName << " > " << lastId;
            query << " ORDER BY 1 LIMIT " << IDS_BATCH_SIZE;
            auto rows = readWithRetries(pool_, query.str());
            if (rows.empty()) {
                break;
            }
            ids.reserve(rows.size());
            for (const auto& row : rows) {
                lastId = row[0].as<ID>();
                if (!existedIds.count(lastId)) {
                    ids.push_back(lastId);
                }
            }

            if (!ids.empty()) {
                count += ids.size();
                auto batchInfo = "batch " + tableName +
                    " [" + std::to_string(ids.front()) + "," +
                           std::to_string(ids.back()) + "]";
                threadPool.push([this, tableName, idName, ids, batchInfo] {
                    safeRunnerQuiet(
                        [&]{ batch<GeomType>(tableName, idName, ids, batchInfo); },
                        batchInfo, fail_);
                });
            }
            if (rows.size() < IDS_BATCH_SIZE) {
                break;
            }
        }
        INFO() << tableName << " pushed, size: "  << count;
    }

    template <typename GeomType>
    void batch(
        const std::string& tableName,
        const std::string& idName,
        const std::vector<ID>& ids,
        const std::string& batchInfo) const
    {
        DEBUG() << batchInfo << " loading ids: " << ids.size();

        auto query =
            "SELECT " + idName + ", ST_AsBinary(shape)"
            " FROM " + schemaName_ + "." + tableName +
            " WHERE " + idName + " IN (" + common::join(ids, ',') + ")"
            " AND " + idName + " BETWEEN " + std::to_string(ids.front()) +
                                   " AND " + std::to_string(ids.back());

        auto rows = readWithRetries(pool_, query);
        ASSERT(rows.size() == ids.size());

        DEBUG() << batchInfo << " calculating";

        std::deque<std::string> values;
        for (const auto& row : rows) {
            auto id = row[0].as<std::string>();
            auto wkb = pqxx::binarystring(row[1]).str();

            for (const auto& isocode : calcIsocodes<GeomType>(id, wkb)) {
                values.push_back(id + ",'" + isocode + "'");
            }
        }
        if (!values.empty()) {
            DEBUG() << batchInfo << " writing";
            auto query =
                "INSERT INTO " + schemaName_ + "." + tableName + "_isocode"
                " VALUES (" + common::join(values, "),(") + ")";
            execCommitWithRetries(pool_, batchInfo, query);
        }
        DEBUG() << batchInfo << " done";
    }

    void checkRegions() const
    {
        Isocodes isocodes;

        auto query =
            "SELECT isocode FROM " + schemaName_ + ".ad "
            " WHERE level_kind=1 AND g_ad_id IS NULL AND isocode IS NOT NULL";

        auto rows = readWithRetries(pool_, query);
        for (const auto& row : rows) {
            isocodes.insert(row[0].as<std::string>());
        }
        INFO() << "ad isocodes: " << isocodes.size();

        auto regions = layer_.regions(boost::none);
        INFO() << "layer regions: " << regions.size();
        for (const auto& region : regions) {
            auto regionId = region.id();
            ASSERT(regionId);
            std::string isocode = region.metaData();
            DATA_REQUIRE(isocodes.count(isocode),
                "unknown isocode for layer region: " << *regionId << " : " << isocode);
        }
    }

    void addIsocodes(
        const std::string& id,
        const geolib3::Point2& point,
        Isocodes& isocodes) const
    {
        auto regions = layer_.regions(point, boost::none);
        if (regions.empty()) {
            if (isocodes.insert(UNKNOWN_ISOCODE_001).second) {
                DEBUG() << "no regions for " << id;
            }
            return;
        }

        Isocodes specialIsocodes;
        bool foundCountries = false;

        for (const auto& region : regions) {
            const char* isocode = region.metaData();
            if (!::isdigit(isocode[0])) {
                foundCountries = true;
                isocodes.insert(isocode);
            } else if (!foundCountries) {
                specialIsocodes.insert(isocode);
            }
        }
        if (!foundCountries) {
            isocodes.insert(specialIsocodes.begin(), specialIsocodes.end());
        }
    }

    template <typename GeomType>
    Isocodes calcIsocodes(
        const std::string& id,
        const std::string& wkb) const;

private:
    pgpool3::Pool& pool_;
    const coverage5::Coverage coverage_;
    const coverage5::Layer& layer_;
    const std::string schemaName_;
    std::shared_ptr<std::atomic<bool>> fail_;
    const std::string coverageFile_;
};

template <>
Isocodes
GeometryProcessor::Impl::calcIsocodes<geolib3::Point2>(
    const std::string& id,
    const std::string& wkb) const
{
    auto geom = geolib3::WKB::read<geolib3::Point2>(wkb);

    Isocodes isocodes;
    addIsocodes(id, geom, isocodes);
    return isocodes;
}

template <>
Isocodes
GeometryProcessor::Impl::calcIsocodes<geolib3::Polyline2>(
    const std::string& id,
    const std::string& wkb) const
{
    auto geom = geolib3::WKB::read<geolib3::Polyline2>(wkb);

    Isocodes isocodes;
    for (const auto& point : geom.points()) {
        addIsocodes(id, point, isocodes);
    }
    return isocodes;
}


GeometryProcessor::GeometryProcessor(
        pgpool3::Pool& pool,
        const std::string& coverageFile,
        const std::string& schemaName,
        std::shared_ptr<std::atomic<bool>> fail)
    : impl_(new Impl(pool, coverageFile, schemaName, std::move(fail)))
{}

GeometryProcessor::~GeometryProcessor() = default;

void
GeometryProcessor::processRdEl(size_t threads) const
{
    impl_->processRdEl(threads);
}

void
GeometryProcessor::processRdJc(size_t threads) const
{
    impl_->processRdJc(threads);
}

void
GeometryProcessor::processEdge(size_t threads) const
{
    impl_->processEdge(threads);
}

void
GeometryProcessor::processNode(size_t threads) const
{
    impl_->processNode(threads);
}

} // namespace maps::wiki::json2ymapsdf::isocode
