#include "constraints.h"

#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/config.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/helpers.h>
#include "pqxx.h"

#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <maps/libs/common/include/file_utils.h>

#include <boost/filesystem.hpp>

#include <atomic>

namespace maps::wiki::json2ymapsdf::ymapsdf::schema {

namespace {

void
createIndex(
    pgpool3::Pool& pool,
    const Constraints::Index& index,
    const std::string& schemaName)
{
    auto tableColumns =
        index.table + " (" + common::join(index.columns, ',') + ")";

    auto query =
        "SET search_path TO " + schemaName + ";\n"
        "CREATE INDEX ON " + tableColumns;

    execCommitWithRetries(pool, "create index " + tableColumns, query);
}

void
validateSqlFile(
    pgpool3::Pool& pool,
    const std::string& sqlFileName,
    const std::string& schemaName)
{
    INFO() << "Validating schema '" << schemaName << "' from file '" << sqlFileName << "'.";

    const auto query = "SET search_path TO " + schemaName + ", public;\n"
        + maps::common::readFileToString(sqlFileName);

    const auto errors = readWithRetries(pool, query);
    for (const auto& row: errors) {
        DATA_ERROR() << row.front().as<std::string>("unknown error");
    }

    DATA_REQUIRE(errors.empty(), "Schema validation '" << schemaName << "' failed.");
}

} // namespace


Constraints::Constraints(
        pgpool3::Pool& pool,
        std::string connStr,
        std::string validationPath)
    : pool_(pool)
    , connStr_(std::move(connStr))
    , validationPath_(std::move(validationPath))
{ }


void
Constraints::init(const std::string& schemaName, const std::string& schemaSql)
{
    dropDBSchema(connStr_, schemaName);
    createDBSchema(connStr_, schemaName, schemaSql);

    for (const auto& row: selectIndexesData(connStr_, schemaName)) {
        indexes_.push_back({
            row.at(column::INDEX_NAME).as<std::string>(),
            row.at(column::TABLE_NAME).as<std::string>(),
            common::parseSqlArray(row.at(column::COLUMN_NAME).as<std::string>()),
            row.at(column::IS_UNIQUE).as<bool>(),
            row.at(column::IS_PRIMARY).as<bool>(),
        });
    }

    for (const auto& row: selectIntegrityData(connStr_, schemaName)) {
        integrityChecks_.push_back({
            row.at(column::CONSTRAINT_NAME).as<std::string>(),
            row.at(column::TABLE_NAME).as<std::string>(),
            row.at(column::COLUMN_NAME).as<std::string>(),
            row.at(column::FOREIGN_TABLE_NAME).as<std::string>(""),
            row.at(column::FOREIGN_COLUMN_NAME).as<std::string>("")
        });
    }

    dropDBSchema(connStr_, schemaName);
}


void
Constraints::createIndexes(const std::string& schemaName, size_t threadCount) const
{
    PROGRESS() << "Creating indexes for schema '" << schemaName << "'.";
    if (!selectIndexesData(connStr_, schemaName).empty()) {
        PROGRESS() << "Indexes for schema '" << schemaName << "' already created.";
        return;
    }

    std::atomic<bool> failed(false);
    auto fail =[&failed](){ failed = true; };
    ThreadPool threadPool(threadCount);

    for (const auto& index: indexes_) {
        threadPool.push(
            safeRunner(
                "Creating index " + index.name + " for schema '" + schemaName + "'",
                fail,
                [&] { createIndex(pool_, index, schemaName); }
            )
        );
    }

    threadPool.shutdown();

    DATA_REQUIRE(!failed, "Creating indexes for schema '" << schemaName << "' failed.");
}

void
Constraints::validateUniqueIndexes(const std::string& schemaName, size_t threadCount) const
{
    PROGRESS() << "Validating unique indexes and primary keys for schema '" << schemaName << "'.";

    std::atomic<bool> failed(false);
    auto fail =[&failed](){ failed = true; };

    ThreadPool threadPool(threadCount);

    for (const auto& index: indexes_) {
        if (!index.isUnique) {
            continue;
        }
        threadPool.push(
            safeRunner(
                "Validating index " + index.name + " for schema '" + schemaName + "'",
                fail,
                [&, index]() {
                    const auto errors = collectUniqueViolations(
                        pool_,
                        schemaName,
                        index.table,
                        index.columns);
                    DATA_REQUIRE(
                        errors.empty(),
                        "Could not create unique constraint " << index.name
                            << " on schema '" << schemaName << "': "
                            << "Keys (" << common::join(errors, ", ") << ") are duplicated."
                    );
                }
            )
        );
    }

    threadPool.shutdown();

    DATA_REQUIRE(!failed, "Validating unique indexes and primary keys for schema '" << schemaName << "' failed.");
}

void
Constraints::validateIntegrityChecks(const std::string& schemaName, size_t threadCount) const
{
    PROGRESS() << "Validating integrity checks for schema '" << schemaName << "'.";

    std::atomic<bool> failed(false);
    auto fail =[&failed](){ failed = true; };

    ThreadPool threadPool(threadCount);

    for (const auto& check: integrityChecks_) {
        threadPool.push(
            safeRunner(
                "Validating integrity checks for schema '" + schemaName + "' "
                    + check.table + "." + check.column + " -> "
                    + check.foreignTable + "." + check.foreignColumn,
                fail,
                [&, check]() {
                    const auto errors = collectForeignKeyViolations(
                        pool_,
                        schemaName,
                        check.table,
                        check.column,
                        check.foreignTable,
                        check.foreignColumn);
                    DATA_REQUIRE(
                        errors.empty(),
                        "Could not apply foreign key on "
                            << schemaName << "." << check.table << "." << check.column
                            << " referenced table '" << check.foreignTable << ": No keys ("
                            <<  common::join(errors, ", ")
                            << ") in " << check.table << " table.";
                    );
                }
            )
        );
    }

    threadPool.shutdown();

    DATA_REQUIRE(!failed, "Applying integrity checks for schema '" << schemaName << "' failed.");
}


void
Constraints::validateSql(const std::string& schemaName, size_t threadCount) const
{
    if (validationPath_.empty()) {
        return;
    }

    PROGRESS() << "Validating schema '" << schemaName << "'.";

    std::atomic<bool> failed(false);
    auto fail =[&failed](){ failed = true; };
    ThreadPool pool(threadCount);

    namespace fs = boost::filesystem;
    fs::path path(validationPath_);
    for (fs::directory_iterator dirIt(path), end; dirIt != end; ++dirIt) {
        std::string sqlFileName = dirIt->path().string();
        if (!sqlFileName.ends_with(".sql")) {
            continue;
        }

        pool.push(
            safeRunner(
                "Validating schema '" + schemaName + "'",
                fail,
                [sqlFileName, &schemaName, this]() {
                    validateSqlFile(pool_, sqlFileName, schemaName);
                }
            )
        );
    }

    pool.shutdown();

    DATA_REQUIRE(!failed, "Validating schema '" << schemaName << "' failed");
}

Constraints
loadConstraints(pgpool3::Pool& pool, const Params& params)
{
    const auto schemaName = params.schema + "_tmp";
    const auto validationPath = config::getPathValidationSqlDir(params.transformCfg);
    const auto schemaSql =
        maps::common::readFileToString(config::getPathSchemaPubSql(params.transformCfg));

    Constraints result(pool, params.connStr, validationPath);
    result.init(schemaName, schemaSql);
    return result;
}

} // namespace maps::wiki::json2ymapsdf::ymapsdf::schema
