#include "schema.h"

#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/config.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/helpers.h>
#include "pqxx.h"

#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/common/include/file_utils.h>

#include <boost/algorithm/string/replace.hpp>
#include <boost/lexical_cast.hpp>

#include <atomic>
#include <memory>

namespace maps::wiki::json2ymapsdf::ymapsdf::schema {

namespace {

void
dropIndexes(const std::string& connStr, const std::string& schemaName)
{
    std::ostringstream query;
    for (const auto& row: selectIndexesData(connStr, schemaName)) {
        const auto indexName = row.at(column::INDEX_NAME).as<std::string>();
        if (row.at(column::IS_PRIMARY).as<bool>()) {
            const auto tableName = row.at(column::TABLE_NAME).as<std::string>();
            query << "ALTER TABLE " << schemaName << "." << tableName << " "
                << "DROP CONSTRAINT IF EXISTS "<< indexName << " CASCADE;\n";
        } else {
            query <<  "DROP INDEX " << schemaName << "." << indexName << ";\n";
        }
    }

    execCommitWithRetries(connStr, "drop indexes " + schemaName, query.str());
}


void
dropIntegrityChecks(const std::string& connStr, const std::string& schemaName)
{
    std::ostringstream query;
    for (const auto& row: selectIntegrityData(connStr, schemaName)) {
        const auto tableName = row.at(column::TABLE_NAME).as<std::string>();
        const auto checkName = row.at(column::CONSTRAINT_NAME).as<std::string>();
        query << "ALTER TABLE " << schemaName << "." << tableName << " "
              << "DROP CONSTRAINT IF EXISTS "<< checkName << " CASCADE;\n";
    }

    execCommitWithRetries(connStr, "drop constraints " + schemaName, query.str());
}

} // namespace


/*****************************************************************************/
/*                        P U B L I C   M E T H O D S                        */
/*****************************************************************************/
Schema::Schema(const std::string& name, const std::string& connStr, const std::string& schemaSql)
    : name_{name}
    , schemaSql_{schemaSql}
{
    INFO() << "Creating schema '" << name << "' from file " << schemaSql;

    dropDBSchema(connStr, name);
    createDBSchema(connStr, name, maps::common::readFileToString(schemaSql));

    initTables(connStr);
    initRelations();

    // Drop unnecessary stuff that was created by the default export SQL.
    dropIntegrityChecks(connStr, name);
    dropIndexes(connStr, name);

    for (const auto& [table, _] : tables_) {
        setUnloggedTable(connStr, name, table);
    }
}


const Table&
Schema::table(const std::string& tableName) const
{
    return *tablePtr(tableName);
}


Table*
Schema::tablePtr(const std::string& tableName) const
{
    const auto& it = tables_.find(tableName);
    if (it != tables_.end()) {
        return &*it->second;
    }
    throw LogicError() << "Unknown table: '" << tableName << "'";
}


bool
Schema::hasTable(const std::string& tableName) const
{
    return tables_.count(tableName);
}


void
Schema::serialize(json::ObjectBuilder& builder) const
{
    builder["YMapsDF schema"] = name();

    for (const auto& kv: tables_) {
        const auto& table = *kv.second;
        builder[table.name()] = [&](json::ObjectBuilder builder) {
            for (const auto& column: table.columns()) {
                builder[column.name()] = [&](json::ObjectBuilder builder) {
                    builder["id"] = column.id();
                    builder["type"] = boost::lexical_cast<std::string>(column.type());
                    builder["is_primary"] = column.isPrimary();
                    builder["is_service"] = column.isService();
                    if (column.foreignPtr() != nullptr) {
                        builder["foreign"] = column.foreignPtr()->name();
                    }
                    if (!column.defaultValue().empty()) {
                        builder["default"] = column.defaultValue();
                    }
                };
            }
        };
    }
}


/*****************************************************************************/
/*                       P R I V A T E   M E T H O D S                       */
/*****************************************************************************/
Table&
Schema::initTable(const std::string& tableName)
{
    const auto& it = tables_.find(tableName);
    if (it != tables_.end()) {
        return *it->second;
    }
    auto table = std::make_shared<Table>(this, tableName);
    tables_[tableName] = table;
    return *table;
}


void
Schema::initTables(const std::string& connStr)
{
    for (const auto& row: selectTablesData(connStr, name())) {
        Table& table = initTable(row.at(column::TABLE_NAME).as<std::string>());
        Column column(
            &table,
            row.at(column::COLUMN_NAME).as<std::string>(),
            row.at(column::DEFAULT_VALUE).as<std::string>(""),
            row.at(column::IS_PRIMARY).as<bool>(false),
            boost::lexical_cast<schema::Type>(row.at(column::COLUMN_TYPE).as<std::string>()),
            row.at(column::FOREIGN_TABLE_NAME).is_null()
                ? nullptr
                : &initTable(row.at(column::FOREIGN_TABLE_NAME).as<std::string>()),
            row.at(column::IS_SERVICE).as<bool>(false));
        table.addColumn(std::move(column));
    }
}


void
Schema::initRelations()
{
    auto addRelation = [this](const Column& master, const Column& slave) {
        relations_.emplace_back(master, slave);
        relations_.emplace_back(slave, master);
    };

    for (const auto& kv: tables_) {
        const auto& table = *kv.second;
        const auto& primary = table.primary();
        auto foreignCols = table.foreign();
        if (!primary.isService()) {
            for (auto foreign: foreignCols) {
                addRelation(primary, *foreign);
            }
        } else if (foreignCols.size() == 2) {
            addRelation(*foreignCols[0], *foreignCols[1]);
        }
    }
}

SchemaPtr
createSchema(const Params& params)
{
    return std::make_unique<Schema>(
        params.schema,
        params.connStr,
        config::getPathSchemaSql(params.transformCfg));
}

namespace {
const std::string SCHEMA_PLACEHOLDER = "%schema%";
const std::string ISOCODES_PLACEHOLDER = "%isocodes%";
} // namespace

std::string
createRegionView(const Params& params, const Region& region)
{
    auto schemaViewName = params.schema + "_" + region.id;
    const auto isocodes = "'" + common::join(region.isoCodes, "','") + "'";


    std::ostringstream query;

    query << "SET search_path TO " << params.schema << ", public;\n";

    const auto schemaFileName = config::getPathSchemaViewSql(params.transformCfg);
    std::ifstream schemaSqlStream(schemaFileName);
    std::string str;
    REQUIRE(schemaSqlStream.rdbuf()->is_open(), "Failed to open file: " << schemaFileName);
    while (getline(schemaSqlStream, str)) {
        boost::replace_all(str, SCHEMA_PLACEHOLDER, schemaViewName);
        boost::replace_all(str, ISOCODES_PLACEHOLDER, isocodes);
        query << str << "\n";
    }

    idManager().roundUp();
    const auto metaId = idManager().uniqueDBID();
    query << "INSERT INTO " << schemaViewName << ".meta (meta_id, type) VALUES (" << metaId << ", 'dataset');\n";

    query << "INSERT INTO " << schemaViewName << ".meta_param (meta_param_id, meta_id, key, value) VALUES \n"
        << "(" << metaId << ", " << metaId << ", "
        << "'region', '" << region.id << "')" << ";\n";

    dropDBSchema(params.connStr, schemaViewName);
    createDBSchema(params.connStr, schemaViewName, query.str());

    return schemaViewName;
}

} // namespace maps::wiki::json2ymapsdf::ymapsdf::schema
