#include "value.h"
#include "schema/pqxx.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/tds/schema.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/log.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/common/data_error.h>

#include <maps/libs/common/include/exception.h>

namespace maps::wiki::json2ymapsdf::ymapsdf {

namespace {

bool
stringToBoolean(const std::string& value)
{
    if (value == tds::VALUE_TRUE || value == ymapsdf::VALUE_TRUE) {
        return true;
    }
    DATA_REQUIRE(value == tds::VALUE_FALSE || value == ymapsdf::VALUE_FALSE,
        "Unable to convert '" << value << "' to bool");
    return false;
}

int64_t
stringToInteger(const std::string& value)
{
    try {
        return std::stol(value);
    } catch (const std::exception& ex) {
        throw TdsDataError() << "Unable to convert '" << value << "' to integer"
            << ": " << ex.what();
    }
}

double
stringToDouble(const std::string& value)
{
    try {
        return std::stod(value);
    } catch (const std::exception& ex) {
        throw TdsDataError() << "Unable to convert '" << value << "' to double"
            << ": " << ex.what();
    }
}

std::string
escapeAndQuote(const std::string& input)
{
    std::string quoted;
    quoted.reserve(input.size() + 2);
    quoted.push_back('\'');
    for (auto c: input) {
        switch(c) {
            case '\'':
            case '\\':
                if (quoted.front() != 'E') {
                    quoted = "E" + quoted;
                    quoted.reserve(input.size() * 2 - quoted.size()  + 4);
                }
                quoted.push_back('\\');
            default:
                quoted.push_back(c);
        }
    }
    quoted.push_back('\'');
    return quoted;
}

inline unsigned char
toHex(unsigned char c) {
    if (c >= 10) {
        ASSERT(c <= 15);
        return c + 'A' - 10;
    }
    return c + '0';
}

std::string
wkb2WkbHex(const std::string& wkb)
{
    std::stringstream wkbHex;
    wkbHex << "ST_SetSRID('";
    for (size_t i = 0; i <  wkb.length();  ++i) {
        wkbHex << toHex(static_cast<unsigned char>(wkb[i]) >> 4)
            << toHex(static_cast<unsigned char>(wkb[i]) & 0x0F);
    }
    wkbHex << "'::geometry, 4326)";
    return wkbHex.str();
}

} // namespace

Value::Value(const schema::Column& column)
    : column_(column)
    , filled_(false)
{ }

Value::Value(std::string& value, const schema::Column& column)
    : column_(column)
    , value_(value)
    , filled_(true)
{ }

Value&
Value::operator = (const Value& value)
{
    REQUIRE(column_.type() == value.column_.type(),
        "Type mismatch: assigning "
        << value.column_.name() << " of type " << value.column_.type()
        << " to "
        << column_.name() << " of type " << column_.type());
    value_ = value.value_;
    filled_ = value.filled_;
    return *this;
}

Value&
Value::operator = (int64_t value)
{
    REQUIRE(column_.type() == schema::Type::Integer,
        "Type mismatch: Unable to assign integer value to column "
        << column_.name() << " of type " << column_.type());
    value_ = std::to_string(value);
    filled_ = true;
    return *this;
}

Value&
Value::operator = (double value)
{
    REQUIRE(column_.type() == schema::Type::Double,
        "Type mismatch: Unable to assign double value to column "
        << column_.name() << " of type " << column_.type());
    value_ = std::to_string(value);
    filled_ = true;
    return *this;
}

Value&
Value::operator = (bool value)
{
    REQUIRE(column_.type() == schema::Type::Bool,
        "Type mismatch: Unable to assign bool value to column "
        << column_.name() << " of type " << column_.type());
    value_ = value ? VALUE_TRUE : VALUE_FALSE;
    filled_ = true;
    return *this;
}

bool
Value::hasDefault() const
{
    return !column_.defaultValue().empty();
}

void
Value::touch()
{
    filled_ = true;
}

Value&
Value::operator = (const std::string& value)
{
    filled_ = true;
    value_.clear();
    if (value.empty()) {
        return *this;
    }
    if (value == VALUE_NULL) {
        if (column_.defaultValue() == VALUE_NULL) {
            value_ = VALUE_NULL;
            return *this;
        }
        REQUIRE(column_.type() == schema::Type::Text,
            "Assigning NULL to NOT NULLABLE " << column_.table().name() << "." << column_.name());
    }

    switch (column_.type()) {
        case schema::Type::Integer:
            *this = stringToInteger(value);
            break;
        case schema::Type::Double:
            *this = stringToDouble(value);
            break;
        case schema::Type::Bool:
            *this = stringToBoolean(value);
            break;
        case schema::Type::Text:
        case schema::Type::Json:
            value_ = escapeAndQuote(value);
            break;
        case schema::Type::ByteArray:
            value_ = "decode('" + value + "', 'base64')";
            break;
        case schema::Type::Geometry:
            value_ = wkb2WkbHex(value);
            break;
    }
    return *this;
}

bool
Value::operator ==(const Value& v) const
{
    return &column_ == &v.column_ && value_ == v.value_;
}

const std::string&
Value::operator *() const
{
    if (value_.empty()) {
        return column().defaultValue();
    }
    return value_;
}

bool
Value::valid() const
{
    return column().isService() || !empty() || !column().defaultValue().empty();
}

} // namespace maps::wiki::json2ymapsdf::ymapsdf
