#include "maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/work/isocode/coverage_builder.h"
#include "maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/lib/work/isocode/geometry_processor.h"
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/json2ymapsdf/tests/fixtures/ymapsdf_fixture.h>

#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/unittest/localdb.h>

#include <boost/filesystem.hpp>
#include <boost/test/unit_test.hpp>

namespace maps {
namespace wiki {
namespace tests {
namespace {

namespace fs = boost::filesystem;


unittest::RandomDatabaseFixture& globalFixture()
{
    static unittest::RandomDatabaseFixture fixture;
    return fixture;
}


class GeometryProcessorFixture: public YmapsdfFixture {
public:
    const size_t THREADS_NUM = 2; // GeometryProcessor does not work with less than 2 threads.
    std::shared_ptr<std::atomic<bool>> fail;

    GeometryProcessorFixture()
        : YmapsdfFixture(
            globalFixture().pool().getMasterConnection(),
            "geometry_processor", // YmapsdfFixture::SCHEMA
            YMAPSDF_EXPORT_FILE)
        , fail(new std::atomic<bool>(false))
    {
        // Remove constraints to make tests easier (no need to fill additional stuff).
        applyQueryToSchema(
            SCHEMA,
            "ALTER TABLE rd_el DROP CONSTRAINT rd_el_f_rd_jc_id_fkey;"
            "ALTER TABLE rd_el DROP CONSTRAINT rd_el_t_rd_jc_id_fkey;"
            "ALTER TABLE rd_el ALTER f_rd_jc_id DROP NOT NULL;"
            "ALTER TABLE rd_el ALTER t_rd_jc_id DROP NOT NULL;"
            "ALTER TABLE edge ALTER f_node_id DROP NOT NULL;"
            "ALTER TABLE edge ALTER t_node_id DROP NOT NULL;"
        );

        prepareAds();
        prepareCoverageFile();
    }

    ~GeometryProcessorFixture() {
        const auto MMS_FILE = TMP_DIR / "ymapsdf.mms.1";
        if (fs::exists(MMS_FILE)) {
            fs::remove(MMS_FILE);
        }
    }

    const std::string coverageFile() const { return coverageFile_.string(); }

    void addGeom(const std::string& category, std::initializer_list<std::string> geoms) {
        size_t id = 1;

        std::string query =
            "INSERT INTO " + category + " (" + category + "_id, shape) VALUES (" +
            common::join(
                geoms,
                [&id](const std::string& geom) {
                    return std::to_string(id++) + ", ST_GeomFromText('" + geom + "', 4326)";
                },
                "), ("
            ) + ")";

        applyQueryToSchema(SCHEMA, query);
    }

    void addIsocodes(const std::string& category, const std::vector<std::pair<size_t, std::string>>& idIsocodes) {
        std::string query =
            "INSERT INTO " + category + "_isocode (" + category + "_id, isocode) VALUES (" +
            common::join(
                idIsocodes,
                [](const std::pair<size_t, std::string>& idIsocode) {
                    return std::to_string(idIsocode.first) + ", '" + idIsocode.second + "'";
                },
                "), ("
            ) + ")";
        applyQueryToSchema(SCHEMA, query);
    }

    const std::string LINE_BETWEEN_AA_AND_BB_BORDERS = "LINESTRING(6 11, 8 5)";
    const std::string LINE_FROM_AA_TO_CC = "LINESTRING(4 7, 4 2)";
    const std::string LINE_FROM_AA_TO_CC_TO_BB = "LINESTRING(5 7, 7 3, 9 7)";
    const std::string LINE_FROM_AA_TO_OUTSIDE = "LINESTRING(1 7, 1 5)";
    const std::string LINE_FROM_OUTSIDE_CROSS_BB = "LINESTRING(7 7, 15 4)";
    const std::string LINE_FROM_OUTSIDE_TO_AA = "LINESTRING(2 5, 2 7)";
    const std::string LINE_FROM_OUTSIDE_TO_AA_HOLE = "LINESTRING(3 5, 3 9)";
    const std::string LINE_IN_AA = "LINESTRING(1 6, 2 6)";
    const std::string LINE_IN_BB = "LINESTRING(11 8, 11 10)";
    const std::string LINE_IN_CC = "LINESTRING(10 10, 12 8)";
    const std::string LINE_IN_OVERLAP_OF_BB_AND_CC = "LINESTRING(9 3, 10 3)";
    const std::string LINE_ON_AA_BORDER = "LINESTRING(2 6, 4 6)";
    const std::string LINE_ON_BORDER_BETWEEN_BB_AND_DD = "LINESTRING(14 8, 14 9)";
    const std::string LINE_OUTSIDE_ADS = "LINESTRING(0 0, 1 1)";
    const std::string POINT_IN_AA = "POINT(1 7)";
    const std::string POINT_IN_AA_HOLE = "POINT(3 9)";
    const std::string POINT_IN_BB = "POINT(13 3)";
    const std::string POINT_IN_CC = "POINT(7 2)";
    const std::string POINT_IN_OVERLAP_OF_BB_AND_CC = "POINT(9 3)";
    const std::string POINT_ON_AA_BORDER = "POINT(4 12)";
    const std::string POINT_ON_BORDER_BETWEEN_BB_AND_DD = "POINT(14 8)";
    const std::string POINT_ON_CC_CORNER_IN_BB = "POINT(11 4)";
    const std::string POINT_OUTSIDE_ADS = "POINT(7 5)";

private:
    const fs::path TMP_DIR = "./";
    fs::path coverageFile_;

    /* The following ADs structure is prepared:
     *    ^
     *    |
     * 12 +-----------+   +-----------+
     *    | 1 AA      |   | 2 BB      |
     * 10 |   +---+   |   |           +-----+
     *    |   |   |   |   |           | 4   |
     *  8 |   +---+   |   |           | DD  |
     *    |           |   |           +-----+
     *  6 +-----------+   |           |
     *    |               |           |
     *  4 |       +-------+-----+     |
     *    |       | 3 CC  |     |     |
     *  2 |       |       +-----+-----+
     *    |       |             |
     *    +-------+-------------+------------>
     *    0   2   4   6   8  10  12  14  16
     */
    void prepareAds() {
        applyQueryToSchema(
            SCHEMA,
            "INSERT INTO ad (ad_id, level_kind, isocode) VALUES"
            "(1, 1, 'AA'),"
            "(2, 1, 'BB'),"
            "(3, 1, 'CC'),"
            "(4, 1, 'DD');"
            "INSERT INTO ad_geom (ad_id, shape) VALUES"
            "(1, ST_GeomFromText('POLYGON((0 6, 6 6, 6 12, 0 12, 0 6), (2 8, 4 8, 4 10, 2 10, 2 8))', 4326)),"
            "(2, ST_GeomFromText('POLYGON((8 2, 14 2, 14 12, 8 12, 8 2))', 4326)),"
            "(3, ST_GeomFromText('POLYGON((3 0, 11 0, 11 4, 3 4, 3 0))', 4326)),"
            "(4, ST_GeomFromText('POLYGON((14 7, 17 7, 17 10, 14 10, 14 7))', 4326));"
        );
    }

    void prepareCoverageFile() {
        coverageFile_ =
            json2ymapsdf::isocode::buildCoverageFile(
                globalFixture().pool(),
                SCHEMA,
                TMP_DIR.string()
            );
    }
};

} // namespace


BOOST_FIXTURE_TEST_SUITE(main, GeometryProcessorFixture)

using json2ymapsdf::isocode::GeometryProcessor;

BOOST_AUTO_TEST_CASE(internalAndExternal) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom("rd_el", {LINE_IN_AA, LINE_IN_BB, LINE_OUTSIDE_ADS});
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("rd_el", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("rd_el", 3, {"001"}));

    addGeom("edge", {LINE_IN_AA, LINE_IN_BB, LINE_OUTSIDE_ADS});
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("edge", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("edge", 3, {"001"}));

    addGeom("rd_jc", {POINT_IN_AA, POINT_IN_AA_HOLE, POINT_OUTSIDE_ADS});
    gp.processRdJc(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_jc", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 2, {"001"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 3, {"001"}));

    addGeom("node", {POINT_IN_AA, POINT_IN_AA_HOLE, POINT_OUTSIDE_ADS});
    gp.processNode(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("node", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("node", 2, {"001"}));
    BOOST_CHECK(hasIsocodes("node", 3, {"001"}));
}


BOOST_AUTO_TEST_CASE(onBorder) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom(
        "rd_el",
        {LINE_ON_AA_BORDER, LINE_BETWEEN_AA_AND_BB_BORDERS, LINE_ON_BORDER_BETWEEN_BB_AND_DD}
    );
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("rd_el", 2, {"AA", "BB"}));
    BOOST_CHECK(hasIsocodes("rd_el", 3, {"BB", "DD"}));

    addGeom(
        "edge",
        {LINE_ON_AA_BORDER, LINE_BETWEEN_AA_AND_BB_BORDERS, LINE_ON_BORDER_BETWEEN_BB_AND_DD}
    );
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("edge", 2, {"AA", "BB"}));
    BOOST_CHECK(hasIsocodes("edge", 3, {"BB", "DD"}));

    addGeom("rd_jc", {POINT_ON_AA_BORDER, POINT_ON_BORDER_BETWEEN_BB_AND_DD, POINT_ON_CC_CORNER_IN_BB});
    gp.processRdJc(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_jc", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 2, {"BB", "DD"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 3, {"BB", "CC"}));

    addGeom("node", {POINT_ON_AA_BORDER, POINT_ON_BORDER_BETWEEN_BB_AND_DD, POINT_ON_CC_CORNER_IN_BB});
    gp.processNode(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("node", 1, {"AA"}));
    BOOST_CHECK(hasIsocodes("node", 2, {"BB", "DD"}));
    BOOST_CHECK(hasIsocodes("node", 3, {"BB", "CC"}));
}


BOOST_AUTO_TEST_CASE(crossBorder) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom(
        "rd_el",
        {
            LINE_FROM_AA_TO_OUTSIDE,
            LINE_FROM_OUTSIDE_TO_AA,
            LINE_FROM_AA_TO_CC,
            LINE_FROM_AA_TO_CC_TO_BB
        }
    );
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"001", "AA"}));
    BOOST_CHECK(hasIsocodes("rd_el", 2, {"001", "AA"}));
    BOOST_CHECK(hasIsocodes("rd_el", 3, {"AA", "CC"}));
    BOOST_CHECK(hasIsocodes("rd_el", 4, {"AA", "BB", "CC"}));

    addGeom(
        "edge",
        {
            LINE_FROM_AA_TO_OUTSIDE,
            LINE_FROM_OUTSIDE_TO_AA,
            LINE_FROM_AA_TO_CC,
            LINE_FROM_AA_TO_CC_TO_BB
        }
    );
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"001", "AA"}));
    BOOST_CHECK(hasIsocodes("edge", 2, {"001", "AA"}));
    BOOST_CHECK(hasIsocodes("edge", 3, {"AA", "CC"}));
    BOOST_CHECK(hasIsocodes("edge", 4, {"AA", "BB", "CC"}));
}


BOOST_AUTO_TEST_CASE(overlap) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom("rd_el", {LINE_IN_OVERLAP_OF_BB_AND_CC});
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"BB", "CC"}));

    addGeom("edge", {LINE_IN_OVERLAP_OF_BB_AND_CC});
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"BB", "CC"}));

    addGeom("rd_jc", {POINT_IN_OVERLAP_OF_BB_AND_CC});
    gp.processRdJc(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_jc", 1, {"BB", "CC"}));

    addGeom("node", {POINT_IN_OVERLAP_OF_BB_AND_CC});
    gp.processNode(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("node", 1, {"BB", "CC"}));
}


BOOST_AUTO_TEST_CASE(dontTouchExistingIsocodes) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom("rd_el", {LINE_IN_AA, LINE_IN_BB, LINE_IN_CC});
    addIsocodes("rd_el", {{1, "XX"}, {3, "ZZ"}});
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"XX"}));
    BOOST_CHECK(hasIsocodes("rd_el", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("rd_el", 3, {"ZZ"}));

    addGeom("edge", {LINE_IN_AA, LINE_IN_BB, LINE_IN_CC});
    addIsocodes("edge", {{1, "XX"}, {3, "ZZ"}});
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"XX"}));
    BOOST_CHECK(hasIsocodes("edge", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("edge", 3, {"ZZ"}));

    addGeom("rd_jc", {POINT_IN_AA, POINT_IN_BB, POINT_IN_CC});
    addIsocodes("rd_jc", {{1, "XX"}, {3, "ZZ"}});
    gp.processRdJc(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_jc", 1, {"XX"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("rd_jc", 3, {"ZZ"}));

    addGeom("node", {POINT_IN_AA, POINT_IN_BB, POINT_IN_CC});
    addIsocodes("node", {{1, "XX"}, {3, "ZZ"}});
    gp.processNode(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("node", 1, {"XX"}));
    BOOST_CHECK(hasIsocodes("node", 2, {"BB"}));
    BOOST_CHECK(hasIsocodes("node", 3, {"ZZ"}));
}


// The current algorithm can't detect crossing borders if all points are out of
// the region.
BOOST_AUTO_TEST_CASE(limitations) {
    GeometryProcessor gp(
        globalFixture().pool(), coverageFile(), SCHEMA, fail);

    addGeom("rd_el", {LINE_FROM_OUTSIDE_TO_AA_HOLE, LINE_FROM_OUTSIDE_CROSS_BB});
    gp.processRdEl(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("rd_el", 1, {"001"}));
    BOOST_CHECK(hasIsocodes("rd_el", 2, {"001"}));

    addGeom("edge", {LINE_FROM_OUTSIDE_TO_AA_HOLE, LINE_FROM_OUTSIDE_CROSS_BB});
    gp.processEdge(THREADS_NUM);
    BOOST_CHECK(hasIsocodes("edge", 1, {"001"}));
    BOOST_CHECK(hasIsocodes("edge", 2, {"001"}));
}


BOOST_AUTO_TEST_SUITE_END()

} // namespace tests
} // namespace wiki
} // namespace maps
