#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/pl2ymapsdf/builder.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/pl2ymapsdf/common.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/pl2ymapsdf/names.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/shp2ymapsdf/database.h>

#include <maps/libs/ymapsdf/include/ad.h>

#include <maps/libs/common/include/exception.h>

#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/serialization.h>

#include <maps/libs/log8/include/log8.h>

#include <yandex/maps/wiki/common/pg_utils.h>

#include <unordered_map>
#include <unordered_set>


namespace maps::wiki::pl2ymapsdf {

namespace {

const std::string ID = "id";
const std::string SHAPE = "shape";
const std::string CENTER = "center";


class Builder
{
public:
    Builder(const std::string& connStr,
            const std::string& srcSchema,
            const std::string& dstSchema)
        : db_(connStr, srcSchema, dstSchema)
    {
        adId_ = db_.maxId("ad");
        nodeId_ = db_.maxId("node");
        faceId_ = db_.maxId("face");
        edgeId_ = db_.maxId("edge");

        nmId_ = db_.maxId("ad_nm", "nm_id");
    }

    void run(const std::unordered_set<std::string>& stages)
    {
        if (stages.count(STAGE_ALL) || stages.count(STAGE_ATD)) {
            processAd();
        }
        db_.commit();
    }

private:
    void insertNames(
        const std::string& table,
        const std::string& key,
        Id id,
        const NamesBase& names)
    {
        auto query = Query()
            .insertInto(table)
            .columns("nm_id, " + key + ", lang, is_local, is_auto, name");
        bool first = true;
        for (const auto& [lang, name] : names.langToName()) {
            auto record = Query()
                << ++nmId_ << ','
                << id << ','
                << db_.quote(lang) << ','
                << (lang == LOCAL_LANG ? "TRUE" : "FALSE") << ','
                << "false" << ','
                << db_.quote(name);
            if (first) {
                query.values(record);
                first = false;
            } else {
                appendValues(query, record);
            }
        }
        db_.write(query);
    }

    void insertFaceEdge(Id faceId, Id edgeId)
    {
        auto query = Query()
            .insertInto("face_edge")
            .columns("face_id, edge_id")
            .values(Query() << faceId << ',' << edgeId);
        db_.write(query);
    }

    Id insertFace(const geolib3::LinearRing2& linearRing)
    {
        auto faceId = ++faceId_;

        auto query = Query()
            .insertInto("face")
            .columns("face_id")
            .values(Query() << faceId);
        db_.write(query);

        geolib3::PointsVector points;
        for (size_t i = 0; i < linearRing.pointsNumber(); ++i) {
            points.push_back(linearRing.pointAt(i));
        }
        points.push_back(linearRing.pointAt(0));

        auto edgeId = insertRingEdge(geolib3::Polyline2(points));
        insertFaceEdge(faceId, edgeId);
        return faceId;
    }

    Id insertRingEdge(const geolib3::Polyline2& polyline)
    {
        auto wkb = geolib3::WKB::toString(polyline);
        auto edgeId = ++edgeId_;
        auto nodeId = insertNode(polyline.pointAt(0));

        auto query = Query()
            .insertInto("edge")
            .columns("edge_id, f_node_id, t_node_id, shape")
            .values(Query()
                << edgeId << ','
                << nodeId << ','
                << nodeId << ','
                << db_.toGeom(wkb));
        db_.write(query);
        return edgeId;
    }

    Id insertEdge(const std::string& wkb)
    {
        auto polyline = geolib3::WKB::read<geolib3::Polyline2>(wkb);
        const auto& points = polyline.points();
        ASSERT(points.size() >= 2);
        auto fNodeId = insertNode(points.front());
        auto tNodeId = insertNode(points.back());

        auto edgeId = ++edgeId_;

        auto query = Query()
            .insertInto("edge")
            .columns("edge_id, f_node_id, t_node_id, shape")
            .values(Query()
                << edgeId << ','
                << fNodeId << ','
                << tNodeId << ','
                << db_.toGeom(wkb));
        db_.write(query);
        return edgeId;
    }

    Id insertNode(const geolib3::Point2& point)
    {
        auto wkb = geolib3::WKB::toString(point);
        return insertNode(wkb);
    }

    Id insertNode(const std::string& wkb)
    {
        auto nodeId = ++nodeId_;
        auto query = Query()
            .insertInto("node")
            .columns("node_id, shape")
            .values(Query()
                << nodeId << ','
                << db_.toGeom(wkb));
        db_.write(query);
        return nodeId;
    }

    void insertFace(
        const std::string& parentTableName,
        Id id,
        const geolib3::LinearRing2& linearRing,
        bool isInterior)
    {
        auto faceId = insertFace(linearRing);
        auto query = Query()
            .insertInto(parentTableName + "_face")
            .columns(parentTableName + "_id, face_id, is_interior")
            .values(Query()
                << id << ','
                << faceId << ','
                << (isInterior ? "TRUE" : "FALSE"));
        db_.write(query);
    }

    void insertPolygon(const std::string& parentTableName, Id id, const geolib3::Polygon2& polygon)
    {
        insertFace(parentTableName, id, polygon.exteriorRing(), false);
        for (size_t i = 0; i < polygon.interiorRingsNumber(); ++i) {
            insertFace(parentTableName, id, polygon.interiorRingAt(i), true);
        }
    }

    void insertPolygons(const std::string& parentTableName, Id id, const geolib3::MultiPolygon2& polygons)
    {
        auto size = polygons.polygonsNumber();
        for (size_t idx = 0; idx < size; ++idx) {
            insertPolygon(parentTableName, id, polygons.polygonAt(idx));
        }
    }

    void insertAd(
        const pqxx::row& row,
        Id id,
        Id parentAdId,
        ymapsdf::ad::LevelKind levelKind,
        const std::string& isocode = EMPTY)
    {
        ASSERT(!parentAdId || adNames_.count(parentAdId));

        auto wkbCenter = pqxx::binarystring(row[CENTER]).str();
        auto idCenter = insertNode(wkbCenter);
        auto wkb = pqxx::binarystring(row[SHAPE]).str();
        auto polygons = geolib3::WKB::read<geolib3::MultiPolygon2>(wkb);

        const auto& names = adNames_.emplace(id, row).first->second;
        names.check(id);

        auto fields = Query() << "ad_id, level_kind";
        auto values = Query() << id << ',' << static_cast<uint32_t>(levelKind);
        if (parentAdId) {
            fields << ", p_ad_id";
            values << ',' << parentAdId;
        }
        if (!isocode.empty()) {
            fields << ", isocode";
            values << ',' << db_.quote(isocode);
        }

        db_.write(Query().insertInto("ad").columns(fields).values(values));

        insertCenter("ad", id, idCenter);
        insertPolygons("ad", id, polygons);
    }

    void insertAdMunicipality(Id id)
    {
        auto query = Query()
            .insertInto("locality")
            .columns("ad_id, municipality")
            .values(Query() << id << ',' << "TRUE");
        db_.write(query);
    }

    void insertCenter(const std::string& parentTableName, Id id, Id nodeId)
    {
        auto query = Query()
            .insertInto(parentTableName + "_center")
            .columns(parentTableName + "_id, node_id")
            .values(Query() << id << ',' << nodeId);
        db_.write(query);
    }

    void processAd();

    Database db_;
    Id adId_ = 0;
    Id nmId_ = 0;
    Id nodeId_ = 0;
    Id faceId_ = 0;
    Id edgeId_ = 0;

    std::unordered_map<Id, Names> adNames_;
};

void Builder::processAd()
{
    auto fields =
        "id::bigint AS " + ID + ","
        "nm_0_pl_l AS name_nat,"
        "level_kind::int,"
        "p_ad_id::bigint,"
        "municipali::int,"
        "ST_AsBinary(ST_Multi(ST_Buffer(ST_Setsrid(geom, 4326), 0))) AS " + SHAPE + ","
        "ST_AsBinary(ST_Setsrid(ST_ClosestPoint(geom,ST_Centroid(geom)),4326)) AS " + CENTER;
    {
        auto query = Query().select(fields).from("pl_ad");
        for (const auto& row : db_.read(query, "atd", ID)) {
            auto id = row[ID].as<Id>(0);
            auto parentId = row["p_ad_id"].as<Id>(0);
            auto levelKind = static_cast<ymapsdf::ad::LevelKind>(row["level_kind"].as<int>());
            insertAd(row, id, parentId, levelKind);
            if (row["municipali"].as<int>() > 0) {
                insertAdMunicipality(id);
            }
        }
    }

    for (const auto& [adId, names] : adNames_) {
        insertNames("ad_nm", "ad_id", adId, names);
    }
}

} // namespace

void buildYMapsDF(
    const std::string& connStr,
    const std::string& srcSchema,
    const std::string& dstSchema,
    const std::string& stages)
{
    INFO() << "DB: " << connStr;
    INFO() << srcSchema << " --> " << dstSchema << " : " << stages;

    auto splitted = common::split(stages, ",");

    Builder builder(connStr, srcSchema, dstSchema);
    builder.run({ splitted.begin(), splitted.end() });

    INFO() << "Done.";
}

} // namespace maps::wiki::pl2ymapsdf
