#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/ymapsdf2json/lib/common.h>
#include <maps/wikimap/mapspro/tools/ymapsdf-conversion/ymapsdf2json/lib/ymapsdf2json.h>

#include <maps/libs/log8/include/log8.h>

#include <iostream>

#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>
#include <boost/algorithm/string/split.hpp>

namespace po = boost::program_options;
namespace ml = maps::log8;

static const std::string OPT_HELP = "help";
static const std::string OPT_CONN = "conn";
static const std::string OPT_SCHEMA = "schema";
static const std::string OPT_OUTPUT_DIR = "output-dir";
static const std::string OPT_FILE_NAME_PATTERN = "file-name-pattern";
static const std::string OPT_OUTPUT_FILE = "output-file";
static const std::string OPT_ID_MODE = "id-mode";
static const std::string OPT_ID_START_FROM = "id-start-from";
static const std::string OPT_OBJECTS_PER_FILE = "objects-per-file";
static const std::string OPT_OBJECTS_PER_DB_READ = "objects-per-db-read";
static const std::string OPT_LOG_LEVEL = "log-level";
static const std::string OPT_MAX_CONNECTIONS = "max-connections";
static const std::string OPT_CATEGORIES = "categories";

static const std::string ID_MODE_RENUMBER = "renumber";
static const std::string ID_MODE_NUMERIC = "numeric"; // synonum for renumber
static const std::string ID_MODE_ORIGINAL = "original";


template <typename T>
void
loadParam(const po::variables_map& vm, const std::string& name, T& param)
{
    if(vm.count(name)) {
        param = vm[name].as<T>();
    }
}

template <>
void
loadParam(const po::variables_map& vm, const std::string& name, bool& param)
{
    param = vm.count(name) > 0 ? true : false;
}

template <>
void
loadParam(const po::variables_map& vm, const std::string& name, maps::IdMode& param)
{
    if (vm.count(name) > 0) {
        auto s = vm[name].as<std::string>();
        if (s == ID_MODE_RENUMBER || s == ID_MODE_NUMERIC) {
            param = maps::IdMode::Renumber;
        } else if (s == ID_MODE_ORIGINAL) {
            param = maps::IdMode::KeepOriginal;
        } else {
            throw maps::LogicError("invalid value, allowed value: " +
                ID_MODE_ORIGINAL + "|" + ID_MODE_RENUMBER);
        }
    }
}

template <>
void
loadParam(const po::variables_map& vm, const std::string& name, std::set<std::string>& param)
{
    if (vm.count(name) > 0) {
        boost::split(param, vm[name].as<std::string>(), [](char c) { return c==','; } );
    }
}


int
main(int argc, char* argv[])
{
    maps::Params params;

    po::options_description desc("Allowed options");
    desc.add_options()
        (OPT_CONN.c_str(), po::value<std::string>(),
            "connection string")
        (OPT_MAX_CONNECTIONS.c_str(), po::value<unsigned int>(),
            "max simultaneous connections, [2, 20], 20 by default")
        (OPT_HELP.c_str(),
            "produce help message")
        (OPT_SCHEMA.c_str(), po::value<std::string>(),
            "database schema")
        (OPT_OBJECTS_PER_FILE.c_str(), po::value<size_t>(),
            "objects per file")
        (OPT_OBJECTS_PER_DB_READ.c_str(), po::value<size_t>(),
            "objects per db read")
        (OPT_OUTPUT_DIR.c_str(), po::value<std::string>(),
            "output directory for files by categories")
        (OPT_OUTPUT_FILE.c_str(), po::value<std::string>(),
            "output file - this option places all the output in single file")
        (OPT_FILE_NAME_PATTERN.c_str(), po::value<std::string>(),
            "boost-format pattern, 1st parameter - category name, 2nd - integer file index")
        (OPT_LOG_LEVEL.c_str(), po::value<std::string>(),
            "log level according to yandex-maps-log (debug|info|warn|error|fatal)")
        (OPT_ID_MODE.c_str(), po::value<std::string>(),
            "ymapsdf -> json id mapping mode\n"
            "Value: original|renumber\n"
            "  original\t - ymapsdf table primary keys used as json ids\n"
            "  renumber\t - generate numeric json ids renumbering each ymapsdf table primary key. "
            "Foreign keys, pointing to absent ids used as is, without renumbering")
        (OPT_CATEGORIES.c_str(), po::value<std::string>(),
            "List of comma separated categories, all categories by default")
        (OPT_ID_START_FROM.c_str(), po::value<size_t>(),
            "initial json id for renumber id mapping mode");

    po::variables_map vm;
    try {
        po::store(po::parse_command_line(argc, argv, desc), vm);
    } catch (po::error& e) {
        std::cerr << e.what() << std::endl;
        return 1;
    }
    po::notify(vm);

    if (argc == 1 || vm.count(OPT_HELP)) {
        std::cerr << "Usage: ymapsdf2json <options>" << std::endl;
        std::cerr << desc << std::endl;
        return 1;
    }

    try {
        loadParam(vm, OPT_CONN, params.db.connString);
        loadParam(vm, OPT_MAX_CONNECTIONS, params.db.maxConnections);
        loadParam(vm, OPT_SCHEMA, params.db.schema);
        loadParam(vm, OPT_OUTPUT_DIR, params.outputDir);
        loadParam(vm, OPT_FILE_NAME_PATTERN, params.fileNamePattern);
        loadParam(vm, OPT_OBJECTS_PER_FILE, params.objectsPerFile);
        loadParam(vm, OPT_OBJECTS_PER_DB_READ, params.objectsPerDbRead);
        loadParam(vm, OPT_OUTPUT_FILE, params.outputFile);
        loadParam(vm, OPT_ID_MODE, params.idMode);
        loadParam(vm, OPT_ID_START_FROM, params.idStartFrom);
        loadParam(vm, OPT_CATEGORIES, params.categories);

        std::string logLevel = "error";
        loadParam(vm, OPT_LOG_LEVEL, logLevel);
        ml::setLevel(logLevel);

        params.validate();
        maps::run(params);
    }
    catch(const maps::Exception &e) {
        std::cerr << "FAIL: " << e.what() << std::endl;
        std::cerr << e << std::endl;
        return 1;
    }
    catch(const std::exception &e) {
        std::cerr << "FAIL: " << e.what() << std::endl;
        return 1;
    }
    return 0;
}
