#include "common.h"
#include "magic_strings.h"
#include "params.h"

#include <yandex/maps/wiki/common/pg_utils.h>
#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/polygon.h>

#include <boost/lexical_cast.hpp>

#include <ctype.h>
#include <iostream>
#include <sstream>
#include <fstream>

namespace {

const std::string DESCRIPTION_VALUE = "YMapsDF data";
const std::string SPLIT_DESCRIPTION_VALUE = "YMapsDF split data";

}

namespace maps {

JsonFileAttributes::JsonFileAttributes(const Params& params, pqxx::transaction_base& tr)
{
    attributes_[jkey::DBHOST] = tr.conn().hostname();
    attributes_[jkey::DBNAME] = tr.conn().dbname();
    attributes_[jkey::SCHEMA] = params.db.schema;
    attributes_[jkey::DESCRIPTION] = params.isSplit()
        ? SPLIT_DESCRIPTION_VALUE
        : DESCRIPTION_VALUE;
}

void JsonFileAttributes::write(json::ObjectBuilder& builder) const
{
    for (const auto& v: attributes_) {
        builder[v.first] = v.second;
    }
}

bool
isPqxxArray(const std::string& values)
{
    return values.size() >= 2
        && values[0] == '{'
        && values[values.length()-1] == '}';
}

std::vector<DBID>
parseDBIDArray(const std::string& values)
{
    std::vector<DBID> ids;

    const auto splitValues = wiki::common::parseSqlArray(values);
    for (const auto& value : splitValues) {
        if (value != STR_NULL) {
            ids.push_back(boost::lexical_cast<DBID>(value));
        }
    }

    return ids;
}

std::vector<DBID>
parseUniqueDBIDArray(const std::string& values)
{
    auto ids = parseDBIDArray(values);

    std::sort(ids.begin(), ids.end());
    ids.erase(std::unique(ids.begin(), ids.end()), ids.end());

    return ids;
}

pqxx::result
workExec(pqxx::transaction_base& tr, const std::string& query)
{
    try {
        ProfileTimer timer;
        auto r = tr.exec(query);
        if (timer.getElapsedTimeNumber() >= 60) {
            std::cerr << "SLOW " << query <<
                " : size=" << r.size() <<
                " : " << timer.getElapsedTime() << std::endl;
        }
        return r;
    }
    catch (const pqxx::failure& ex) {
        throw RuntimeError() << "pqxx exec error: " << ex.what() << "\n" << query;
    }
    catch (const std::exception& ex) {
        throw RuntimeError() << "std error: " << ex.what() << "\n" << query;
    }
}

std::vector<DBID>
loadIds(pqxx::transaction_base& tr, const std::string& query)
{
    std::vector<DBID> ids;
    auto data = maps::workExec(tr, query);
    ids.reserve(data.size());
    for (size_t i = 0; i < data.size(); ++i) {
        ids.push_back(data[i][0].as<DBID>());
    }
    return ids;
}

void setSearchPath(pqxx::transaction_base& tr, const std::string& schema)
{
    const std::string setPath = "SET search_path TO "
        + (schema.empty() ? "public" : schema + ",public" );
    workExec(tr, setPath);
}

}//namespace maps
