#include "file_writer.h"
#include "tasks.h"
#include "params.h"
#include "magic_strings.h"

#include <boost/filesystem.hpp>
#include <boost/format.hpp>

#include <fstream>
#include <atomic>

namespace bf = boost::filesystem;

namespace maps {

Writer::~Writer() {}

namespace {

class CategoryWriter {
public:
    CategoryWriter(const Category& category)
        : category_(category), nextFileIndex_(0)
    {}

    void put(TaskContext& context, JsonChunk chunk)
    {
        std::unique_lock<std::mutex> lock(mutex_);
        queue_.insert(queue_.end(), chunk.objects.begin(), chunk.objects.end());
        createTasks(context, false);
    }

    void finish(TaskContext& context)
    {
        createTasks(context, true);
    }

private:
    void createTasks(TaskContext& context, bool finished);
    void createTask(TaskContext& context, std::vector<JsonObject>);

private:
    const Category& category_;
    std::mutex mutex_;
    std::deque<JsonObject> queue_;
    size_t nextFileIndex_;
};

class SingleFileWriter: public Writer {
public:
    void put(TaskContext& context, JsonChunk chunk) override
    {
        createTaskLocked(context);
        queue_.push(std::move(chunk));
    }

    void finish(TaskContext&) override
    {
        queue_.finish();
    }

private:
    void createTaskLocked(TaskContext& context)
    {
        std::call_once(onceFlag, &SingleFileWriter::createTask, this, std::ref(context));
    }

    void createTask(TaskContext& context)
    {
        context.fileWritePool.push(makeTask<WriteSingleFileTask>(context, queue_));
    }

    mw::ThreadedQueue<JsonChunk> queue_;
    std::once_flag onceFlag;
};

class FileWriter: public Writer {
public:
    FileWriter()
    {
        for (const Category& category: CATEGORIES) {
            categoryWriters_.insert(std::make_pair(
                    category.name(),
                    std::unique_ptr<CategoryWriter>(new CategoryWriter(category))));
        }
    }

    void put(TaskContext& context, JsonChunk chunk) override
    {
        auto it = categoryWriters_.find(chunk.category.name());
        REQUIRE(it != categoryWriters_.end(), "no writer for category '" << chunk.category.name() << "'");
        it->second->put(context, std::move(chunk));
    }

    void finish(TaskContext& context) override
    {
        for (auto& categoryWriter: categoryWriters_) {
            categoryWriter.second->finish(context);
        }
    }

private:
    std::unordered_map<std::string, std::unique_ptr<CategoryWriter>> categoryWriters_;
};

void CategoryWriter::createTasks(TaskContext& context, bool finished)
{
    const size_t objectsPerFile = context.params.objectsPerFile;
    if ((queue_.size() < objectsPerFile) && !finished)
        return;
    size_t thisFileSize = std::min(queue_.size(), objectsPerFile);
    while (queue_.size()) {
        std::vector<JsonObject> tmp;
        tmp.reserve(thisFileSize);
        for (size_t thisFileLeft = thisFileSize; thisFileLeft; --thisFileLeft) {
            tmp.push_back(std::move(queue_.front()));
            queue_.pop_front();
        }
        createTask(context, std::move(tmp));
        if ((queue_.size() < objectsPerFile) && !finished)
            return;
        thisFileSize = std::min(queue_.size(), objectsPerFile);
    }
}

void CategoryWriter::createTask(TaskContext& context, std::vector<JsonObject> objects)
{
    bf::path base(context.params.outputDir);
    bf::path filePath = base / (
        boost::format(context.params.fileNamePattern)
        % category_.name()
        % nextFileIndex_).str();
    context.fileWritePool.push(
        makeTask<WriteFileTask>(context, filePath, std::move(objects)));
    ++nextFileIndex_;
}

} // namespace

std::unique_ptr<Writer> Writer::make(const Params& params) {
    if (params.isSplit()) {
        return std::unique_ptr<Writer>(new FileWriter());
    } else {
        return std::unique_ptr<Writer>(new SingleFileWriter());
    }
}

} // namespace maps
