#include "id_manager.h"
#include "magic_strings.h"

#include <mutex>
#include <atomic>
#include <condition_variable>

namespace maps {

namespace {

class IdManager {
public:
    IdManager()
    {
        init();
    }

    void init(IdMode idMode = IdMode::Renumber, DBID idStartFrom = 0)
    {
        completed_ = false;
        tidMap_.clear();
        newIds_.clear();
        idMode_ = idMode;
        maxId_ = idStartFrom;
    }

    DBID convertId(const Table& table, DBID oId)
    {
        std::unique_lock<std::mutex> lock(mutex_);
        while (true) {
            const auto& idMap = tidMap_[&table];
            auto result = idMap.find(oId);
            if (result != idMap.end()) {
                return result->second;
            } else if (completed_) {
                return oId;
            }
            completedCondition_.wait_for(lock, std::chrono::seconds(1));
        }
    }

    void registerId(const Table& table, DBID oId)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        REQUIRE(!completed_, "Unable to register id: registration is completed");

        DBID nId = newId(oId);
        bool newIsUnique = newIds_.insert(nId).second;
        bool oldIsUnique = tidMap_[&table].insert({oId, nId}).second;
        REQUIRE(newIsUnique && oldIsUnique, "Unable to register id " << table.name << ":" << oId << " twice");
    }

    void completeRegistration()
    {
        std::lock_guard<std::mutex> lock(mutex_);
        completed_ = true;
        completedCondition_.notify_all();
    }

    void storeIdMapping(pqxx::transaction_base& work, size_t batchSize)
    {
        work.exec("TRUNCATE TABLE ymapsdf_tds;");
        std::vector<std::tuple<const Table*, DBID, DBID>> idMapBatch;
        idMapBatch.reserve(batchSize);
        for (const auto& it: tidMap_) {
            const auto* table = it.first;
            const auto& idMap = it.second;
            for (const auto& jt: idMap) {
                idMapBatch.emplace_back(std::make_tuple(table, jt.first, jt.second));
                if (batchSize > 0 && idMapBatch.size() >= batchSize) {
                    work.exec(printIdMapBatch(idMapBatch));
                    idMapBatch.clear();
                    idMapBatch.reserve(batchSize);
                }
            }
        }
        work.exec(printIdMapBatch(idMapBatch));
    }

    std::string printIdMapBatch(
        const std::vector<std::tuple<const Table*, DBID, DBID>>& idMapBatch)
    {
        std::ostringstream os;
        os << "INSERT INTO ymapsdf_tds (table_name, ymapsdf_id, tds_id) VALUES" << std::endl;
        for (const auto& record: idMapBatch) {
            os << "('" << std::get<0>(record)->name << "', "
                << std::get<1>(record) << ", "
                << std::get<2>(record) << ")"
                << (&record == &idMapBatch.back() ? ";" : ",")
                << std::endl;
        }
        return os.str();
    }

    DBID maxId() const
    {
        std::unique_lock<std::mutex> lock(mutex_);
        while (!completed_) {
            completedCondition_.wait(lock);
        }
        return maxId_;
    }


private:
    DBID newId(DBID oId)
    {
        switch (idMode_) {
            case IdMode::Renumber :
                return maxId_++;
            case IdMode::KeepOriginal :
                maxId_ = std::max(maxId_, oId);
                return oId;
            default:
                throw LogicError() << "IdMode not implemented: "
                    << static_cast<int>(idMode_);
        }
    }

private:
    typedef std::map<const Table*, std::unordered_map<DBID,DBID>> TIdMap;

private:
    DBID maxId_;
    TIdMap tidMap_;
    std::unordered_set<DBID> newIds_;
    IdMode idMode_;

    bool completed_;
    mutable std::mutex mutex_;
    mutable std::condition_variable completedCondition_;
};


IdManager idManager;

} // namespace

void
initIdManager(const Params& params)
{
    idManager.init(params.idMode, params.idStartFrom);
}

void
registerIds(const Table& table, const std::vector<DBID>& ids)
{
    const auto& baseTable = table.base();
    for(auto oId: ids) {
        idManager.registerId(baseTable, oId);
    }
}

std::string
convertId(const Table& table, DBID id)
{
    return std::to_string(idManager.convertId(table.base(), id));
}

void
completeIdRegistration(pqxx::transaction_base& work, size_t batchSize)
{
    idManager.completeRegistration();
    idManager.storeIdMapping(work, batchSize);
}

DBID
maxId()
{
    return idManager.maxId();
}

} // namespace maps
