#include "attribute_dumper.h"
#include "common.h"
#include "magic_strings.h"
#include "relation_dumper.h"

#include <set>
#include <string>
#include <vector>

namespace maps {

namespace {

void dumpReleation(
    json::ArrayBuilder& builder,
    const Relation& rel,
    DBID id,
    const StringMap& attributes)
{
    if (!id) {
        return;
    }
    const auto& jkeyKind =
        rel.direction == RelationDirection::ToMaster
        ? jkey::MASTER
        : jkey::SLAVE;

    builder << [&](json::ObjectBuilder builder) {
        builder[jkeyKind] = convertId(rel.toTable, id);
        builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
            AttributeDumper ad(ns::REL, builder);
            ad.dump(attribute::MASTER, rel.master);
            ad.dump(attribute::SLAVE, rel.slave);
            ad.dump(attribute::ROLE, rel.role);
            for (const auto& p : attributes) {
                ad.dump(p.first, p.second);
            }
        };
    };
}

} // namespace

RelationDumper::RelationDumper(
        json::ArrayBuilder& builder,
        const pqxx::row& tuple)
    : builder_(builder)
    , tuple_(tuple)
{ }

void
RelationDumper::dump(const Relation& relation, DBID id, const StringMap& attributes)
{
    dumpReleation(builder_, relation, id, attributes);
}

void
RelationDumper::dump(const Relation& relation, const std::string& idName, const StringMap& attributes)
{
    std::vector<DBID> ids;

    auto idString = tuple_.at(idName).as<std::string>(STR_EMPTY_ARRAY);
    if (isPqxxArray(idString)) {
        ids = parseUniqueDBIDArray(idString);
    } else {
        ids = { boost::lexical_cast<DBID>(idString) };
    }

    for (const auto& id : ids) {
        dump(relation, id, attributes);
    }
}

void
RelationDumper::dump(const Relation& relation)
{
    const auto& idName = relation.toTable.id;
    dump(relation, idName);
}

} // namespace maps
