#include "../categories.h"
#include "../magic_strings.h"
#include "../attribute_dumper.h"
#include "../relation_dumper.h"
#include "../ft_type.h"

#include <yandex/maps/wiki/common/pg_utils.h>

#include <boost/algorithm/string/join.hpp>

namespace maps {

namespace {
const std::string AD_ID = "ad_id";
const std::string CENTER = "center";
const std::string P_AD_ID = "p_ad_id";
const std::string G_AD_ID = "g_ad_id";
const std::string DISP_CLASS = "disp_class";
const std::string SEARCH_CLASS = "search_class";
const std::string EXCL_AD_IDS = "excl_ad_ids";
const std::string ISOCODE = "isocode";
const std::string SUBCODE = "subcode";
const std::string IS_INTERIOR = "is_interior";
const std::string LEVEL_KIND = "level_kind";
const std::string LOCALITY = "locality";
const std::string POPULATION = "population";
const std::string CAPITAL = "capital";
const std::string POPULATION_IS_APPROXIMATED = "population_is_approximated";
const std::string TOWN = "town";
const std::string MUNICIPALITY = "municipality";
const std::string INFORMAL = "informal";
const std::string AD_RECOGNITION = "recognition";
const std::string FACE_IDS = "face_ids";
const std::string SUBST_FC_IDS = "subst_fc_ids";
const std::string IS_VIRTUAL = "is_virtual";
const int SEARCH_CLASS_VIRTUAL = 9;

std::string
recognizedBy(const pqxx::row& tuple)
{
    auto recogn = wiki::common::parseSqlArray(
        tuple.at(AD_RECOGNITION).as<std::string>(STR_EMPTY_ARRAY));
    recogn.erase(std::remove(recogn.begin(), recogn.end(), STR_NULL), recogn.end());
    std::sort(recogn.begin(), recogn.end());
    recogn.erase(std::unique(recogn.begin(), recogn.end()), recogn.end());
    return boost::algorithm::join(recogn, ARRAY_DELIM);
}

bool
isVirtual(const pqxx::row& tuple)
{
    const auto searchClass = tuple.at(SEARCH_CLASS);
    return !searchClass.is_null() && searchClass.as<short>() == SEARCH_CLASS_VIRTUAL;
}

bool
isSubst(const pqxx::row& tuple)
{
    return !tuple.at(G_AD_ID).is_null();
}

const std::string&
categoryName(const pqxx::row& tuple)
{
    return isSubst(tuple) ? category::AD_SUBST : category::AD;
}

void
dumpAttributes(AttributeDumper& ad, const pqxx::row& tuple)
{
    ad.dump<int>(LEVEL_KIND, tuple);
    ad.dump<int>(DISP_CLASS, tuple);
    ad.dump<std::string>(ISOCODE, tuple);
    ad.dump<std::string>(SUBCODE, tuple);
    ad.dump<int>(POPULATION, tuple);
    ad.dump<int>(CAPITAL, tuple);
    ad.dump<bool>(TOWN, tuple);
    ad.dump<bool>(MUNICIPALITY, tuple);
    ad.dump<bool>(INFORMAL, tuple);
    ad.dump<bool>(POPULATION_IS_APPROXIMATED, tuple);
    ad.dump(AD_RECOGNITION, recognizedBy(tuple));

    ad.dump(IS_VIRTUAL, isVirtual(tuple));
}

void
dumpRelations(RelationDumper& rd, const pqxx::row& tuple)
{
    if (!isSubst(tuple)) {
        rd.dump(relation::AD_CHILD, P_AD_ID);
        rd.dump(relation::AD_EXCLUSION, EXCL_AD_IDS);
        rd.dump(relation::AD_CENTER, CENTER);
        rd.dump(relation::AD_PART, FACE_IDS);
    } else {
        rd.dump(relation::AD_AD_SUBST_CHILD, P_AD_ID);
        rd.dump(relation::AD_SUBST_SUBSTITUTION, G_AD_ID);
        rd.dump(relation::AD_SUBST_EXCLUSION, EXCL_AD_IDS);
        rd.dump(relation::AD_SUBST_PART, SUBST_FC_IDS);
    }
}

} // namespace

struct AdCategory: public Category {
    const Table& table() const override
    {
        return table::AD;
    }

    std::string loadRowsSqlTemplate() const override
    {
        return "WITH paged_ad AS (SELECT * FROM ad WHERE ad_id IN %1%) "
            "SELECT"
            " paged_ad." + AD_ID + " " + ID + "," +
            " paged_ad." + P_AD_ID + "," +
            " paged_ad." + G_AD_ID + "," +
            " paged_ad." + LEVEL_KIND + "," +
            " paged_ad." + DISP_CLASS + "," +
            " paged_ad." + SEARCH_CLASS + "," +
            " paged_ad." + ISOCODE + "," +
            " paged_ad." + SUBCODE + "," +
            " locality." + POPULATION + ", " +
            " locality." + TOWN + ", " +
            " locality." + MUNICIPALITY + ", " +
            " locality." + INFORMAL + ", " +
            " locality." + POPULATION_IS_APPROXIMATED + ", " +
            " locality." + CAPITAL + ", " +
            " (SELECT ad_center.node_id FROM ad_center" +
                " WHERE ad_center.ad_id = paged_ad.ad_id) " + CENTER + "," +
            " (SELECT array_agg(ad_excl.e_ad_id) FROM ad_excl" +
                " WHERE ad_excl.t_ad_id = paged_ad.ad_id) " + EXCL_AD_IDS + "," +
            " (SELECT array_agg(ad_recognition.isocode) FROM ad_recognition" +
                " WHERE ad_recognition.ad_id = paged_ad.ad_id) " + AD_RECOGNITION + "," +

            " (SELECT array_agg(ad_face.face_id) FROM ad_face" +
                " WHERE ad_face.ad_id = paged_ad.ad_id) " + FACE_IDS + "," +

            " (SELECT array_agg(ad_face_patch.face_id) FROM ad_face_patch" +
                " WHERE ad_face_patch.ad_id = paged_ad.ad_id) " + SUBST_FC_IDS + " " +

            "FROM paged_ad "
            "LEFT JOIN locality ON (locality.ad_id = paged_ad.ad_id)";
    }

    void tupleToJson(
        json::ObjectBuilder& builder,
        const pqxx::row& tuple) const override
    {
        builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
            AttributeDumper ad(categoryName(tuple), builder);
            ad.dumpCategory();
            dumpAttributes(ad, tuple);
        };

        builder[jkey::RELATIONS] = [&](json::ArrayBuilder builder) {
            RelationDumper rd(builder, tuple);
            dumpRelations(rd, tuple);
        };
    }
};

DEFINE_CATEGORY_OBJECT(Ad, AD);

} // namespace maps
