#include "common_nm.h"
#include "../magic_strings.h"
#include "../attribute_dumper.h"
#include "../relation_dumper.h"

#include <maps/libs/locale/include/convert.h>

#include <boost/algorithm/string.hpp>

namespace maps {

namespace {

const std::string NM_ID = "nm_id";
const std::string NAME = "name";
const std::string NAME_TYPE = "name_type";
const std::string LANG = "lang";
const std::string SCRIPT = "script";
const std::string REGION = "region";
const std::string VARIANT = "variant";
const std::string IS_LOCAL = "is_local";
const std::string IS_AUTO = "is_auto";

const std::string&
getNmRole(const pqxx::row& tuple)
{
    switch (tuple.at(NAME_TYPE).as<unsigned>()) {
        case 0: return role::OFFICIAL;
        case 1: return role::RENDER_LABEL;
        case 2: return role::ADDRESS_LABEL;
        case 3: return role::SHORT;
        case 4: return role::SYNONYM;
        case 5: return role::OLD;
        default: throw RuntimeError() << "Object " << tuple.at(ID).as<DBID>()
            << ": Unknown name_type '"
            << tuple.at(NAME_TYPE).as<std::string>() << "'";
    }
}

std::string
getNmLanguage(const pqxx::row& tuple)
{
    locale::Language language;

    language.lang = locale::to<locale::Lang>(tuple.at(LANG).as<std::string>());
    if (!tuple.at(SCRIPT).is_null()) {
        language.script = locale::to<locale::Script>(tuple.at(SCRIPT).as<std::string>());
    }
    if (!tuple.at(REGION).is_null()) {
        language.region = locale::to<locale::Region>(tuple.at(REGION).as<std::string>());
    }
    if (!tuple.at(VARIANT).is_null()) {
        language.variant = locale::to<locale::Langvariant>(tuple.at(VARIANT).as<std::string>());
    }
    return locale::toString(language);
}

std::string
getName(const pqxx::row& tuple)
{
    std::string name = tuple.at(NAME).as<std::string>();
    boost::algorithm::trim(name);
    return name;
}

std::string
nmRecordFields(const Table& toTable)
{
    return NM_ID + " " + ID + ", " +
        NAME + ", " +
        LANG + ", " +
        NAME_TYPE + ", " +
        SCRIPT + ", " +
        REGION + ", " +
        VARIANT + ", " +
        IS_LOCAL + ", " +
        toTable.id + ", " +
        IS_AUTO;
}

} // namespace

std::string
NmCategory::category(const pqxx::row&) const
{
    return name();
}

std::string
NmCategory::masterCategory(const pqxx::row&) const
{
    return masterTable().name;
}

std::string
NmCategory::extraSqlFields() const
{
    return std::string();
}

std::string
NmCategory::loadRowsSqlTemplate() const
{
    const auto extraFields = extraSqlFields();
    return
        "SELECT " + nmRecordFields(masterTable().base()) + " "
         + (extraFields.empty() ? "" : ", ") + extraFields + " "
        "FROM " + table().name + " "
        "WHERE nm_id IN %1%";
}

void
NmCategory::tupleToJson(
    json::ObjectBuilder& builder,
    const pqxx::row& tuple) const
{
    const auto nameCategory = category(tuple);

    builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
        AttributeDumper ad(nameCategory, builder);
        ad.dumpCategory();

        ad.dump(NAME, getName(tuple));
        ad.dump(LANG, getNmLanguage(tuple));
        ad.dump<bool>(IS_LOCAL, tuple);
        ad.dump<bool>(IS_AUTO, tuple);
    };

    builder[jkey::RELATIONS] = [&](json::ArrayBuilder builder) {
        Relation relation = {
            masterCategory(tuple),
            getNmRole(tuple),
            nameCategory,
            masterTable(),
            RelationDirection::ToMaster
        };
        RelationDumper rd(builder, tuple);
        rd.dump(relation);
    };
}

} // namespace maps
