#include "../categories.h"
#include "../magic_strings.h"
#include "../attribute_dumper.h"
#include "../relation_dumper.h"

namespace maps {

namespace {
const std::string ACCESS_ID = "access_id";
const std::string COND_DT_IDS = "cond_dt_ids";
const std::string COND_ID = "cond_id";
const std::string COND_TYPE = "cond_type";
const std::string RD_EL_FROM_ID = "rd_el_from_id";
const std::string RD_EL_TO_IDS = "rd_el_to_ids";
const std::string RD_JC_ID = "rd_jc_id";
const std::string SEQ_NUM = "seq_num";
} // namespace

struct CondCategory: public Category {
    const Table& table() const override
    {
        return table::COND;
    }

    std::string loadRowsSqlTemplate() const override
    {
        return "WITH paged_cond AS (SELECT * FROM cond WHERE cond_id IN %1%) "
            "SELECT"
            " paged_cond." + COND_ID + " " + ID + "," +
            " paged_cond." + COND_TYPE + "," +
            " paged_cond." + ACCESS_ID + "," +
            " (SELECT rd_jc_id FROM cond_rd_seq" +
                " WHERE cond_rd_seq.cond_seq_id = paged_cond.cond_seq_id" +
                " AND cond_rd_seq.seq_num = 0) " + RD_JC_ID + "," +
            " (SELECT rd_el_id FROM cond_rd_seq" +
                " WHERE cond_rd_seq.cond_seq_id = paged_cond.cond_seq_id" +
                " AND cond_rd_seq.seq_num = 0) " + RD_EL_FROM_ID + "," +
            " (ARRAY(SELECT rd_el_id FROM cond_rd_seq" +
                " WHERE cond_rd_seq.cond_seq_id = paged_cond.cond_seq_id" +
                " AND cond_rd_seq.seq_num > 0 ORDER BY seq_num ASC)) "+ RD_EL_TO_IDS + ", " +
            " ARRAY(SELECT cond_dt_id FROM cond_dt WHERE cond_dt.cond_id = paged_cond.cond_id) " + COND_DT_IDS + " " +
            "FROM paged_cond "
            "LEFT JOIN cond_rd_seq ON "
            "    (cond_rd_seq.cond_seq_id = paged_cond.cond_seq_id "
            "     AND cond_rd_seq.seq_num > 0) "
            "GROUP BY"
            " paged_cond.cond_id,"
            " paged_cond.cond_seq_id,"
            " paged_cond.cond_type,"
            " paged_cond.access_id";
    }

    void tupleToJson(
        json::ObjectBuilder& builder,
        const pqxx::row& tuple) const override
    {
        builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
            AttributeDumper ad(name(), builder);
            ad.dumpCategory();
            ad.dump<int>(ACCESS_ID, tuple);
            ad.dump<int>(COND_TYPE, tuple);
        };

        builder[jkey::RELATIONS] = [&](json::ArrayBuilder builder) {
            RelationDumper rd(builder, tuple);

            rd.dump(relation::COND_APPLIED_TO, COND_DT_IDS);
            rd.dump(relation::COND_FROM, RD_EL_FROM_ID);
            rd.dump(relation::COND_VIA, RD_JC_ID);

            const auto rdElIds = parseDBIDArray(tuple.at(RD_EL_TO_IDS).as<std::string>());
            for (size_t i = 0; i < rdElIds.size(); ++i) {
                StringMap attributes = {{SEQ_NUM, std::to_string(i)}};
                rd.dump(relation::COND_TO, rdElIds[i], attributes);
            }
        };
    }
};

DEFINE_CATEGORY_OBJECT(Cond, COND);

} // namespace maps
