#include "../categories.h"
#include "../magic_strings.h"
#include "../attribute_dumper.h"
#include "../relation_dumper.h"
#include "../ft_type.h"
#include "../ft_helper.h"

#include <maps/libs/json/include/value.h>
#include <boost/algorithm/string.hpp>

#include <iostream>
#include <set>

namespace maps {

namespace {
const std::string FT_TYPE_ID = "ft_type_id";
const std::string DISP_CLASS = "disp_class";
const std::string POINT_SHAPE = "point_shape";
const std::string POLYGON_SHAPE = "polygon_shape";
const std::string LINE_SHAPE = "line_shape";
const std::string SOURCE_ID = "source_id";
const std::string NAME = "name";
const std::string IMPORT_SOURCE_ID = "import_source_id";
const std::string IMPORT_SOURCE = "import_source";
const std::string IMPORT_SOURCE_ORG = "org";
const std::string BUSINESS_ID = "business_id";
const std::string ZIPCODE_ADDR_IDS = "zipcode_addr_ids";

const std::string POI_PREFIX = "poi_";
const std::string POI_ATTR_PREFIX = "poi";
const std::string PARENT_PREFIX = "parent_";
const std::string FT_ATTR_JSON = "ft_attr_json";
const std::string FT_FT_JSON_SLAVE = "ft_ft_json_slave";
const std::string FT_FT_JSON_MASTER = "ft_ft_json_master";
const std::string FT_FT_JSON_ROLE = "role";
const std::string FT_FT_JSON_FT_TYPE_ID = "ft_type_id";
const std::string FT_FT_JSON_NOT_POINT = "not_point";
const std::string FT_FT_JSON_FALSE = "false";

const std::string ROLE_ZIPCODE_FOR_ADDR = "2";

std::string
fixEmbededJSON(std::string&& jsonString)
{
    boost::replace_all(jsonString, "\\", "");
    boost::replace_all(jsonString, "\"{", "{");
    boost::replace_all(jsonString, "}\"", "}");
    return std::move(jsonString);
}
} // namepsace

struct FtCategory: public Category {
    const Table& table() const override
    {
        return table::FT;
    }

    std::string loadIdsSql() const override
    {
        return "SELECT ft_id FROM ft "
            "WHERE NOT (" + isTrivialSql("ft.ft_id") + ") "
            " AND (SELECT value FROM ft_attr WHERE ft_attr.ft_id = ft.ft_id  AND key = '_skip_import') IS NULL";
    }

    std::string loadRowsSqlTemplate() const override
    {
        return "WITH paged_ft AS (SELECT * FROM ft WHERE ft.ft_id IN %1%) "
            "SELECT "
            "paged_ft.ft_id " + ID  + ", "
            "paged_ft." + DISP_CLASS + ", "
            "source_type." + NAME + ", "
            "ft_source." + SOURCE_ID + ","

            "(SELECT ST_AsGeoJson(shape) FROM node, ft_center "
                "WHERE ft_center.ft_id = paged_ft.ft_id "
                "AND ft_center.node_id = node.node_id"
                ") " + POINT_SHAPE + "," +

            "(SELECT ST_AsGeoJson(shape) FROM edge, ft_edge "
                "WHERE ft_edge.ft_id = paged_ft.ft_id "
                "AND ft_edge.edge_id = edge.edge_id "
                "AND paged_ft.ft_type_id IN (" + lineFtTypes() + ")"
                ") " + LINE_SHAPE + "," +

            "(SELECT "
                "ST_AsGeoJson((ST_Dump(ST_BuildArea(ST_Collect(edge.shape)))).geom) "
                "FROM ft_face, face_edge, edge "
                "WHERE ft_face.ft_id = paged_ft.ft_id "
                "AND edge.edge_id = face_edge.edge_id "
                "AND face_edge.face_id = ft_face.face_id "
                "AND paged_ft.ft_type_id IN (" + polygonFtTypes() + ")"
                ") " + POLYGON_SHAPE + ", " +

            ftSql("paged_ft.p_ft_id", PARENT_PREFIX) + ", " +

            ftSql("paged_ft.ft_id") + ", "
            "(SELECT json_object(array_agg(key) , array_agg(value))"
            "       FROM ft_attr "
            "       WHERE ft_attr.ft_id = paged_ft.ft_id) ft_attr_json, "
            //Next 2 parts of query will produce json with embeded part like string
            //{"1255" : "{\"role\" : \"previous\", \"ft_type_id\" : \"2206\", \"not_point\" : \"true\"}"}"}
            //need to fix unqoute it
            //1st  relation with slave
            "(SELECT json_object(array_agg(text(slave_ft_id)), "
            " array_agg("
            "    text(json_build_object('role', role, 'ft_type_id', text(ft.ft_type_id), 'not_point', text(node_id IS NULL)))))"
            "       FROM ft_ft, ft LEFT JOIN node ON node.node_id=ft.ft_id"
            "       WHERE ft.ft_id=ft_ft.slave_ft_id AND  "
            "       ft_ft.master_ft_id = paged_ft.ft_id) ft_ft_json_slave, "
            //2st  relation with master
            "(SELECT json_object(array_agg(text(master_ft_id)), "
            " array_agg("
            "    text(json_build_object('role', role, 'ft_type_id', text(ft.ft_type_id), 'not_point', text(node_id IS NULL)))))"
            "       FROM ft_ft LEFT JOIN ft_attr ON ft_ft.master_ft_id = ft_attr.ft_id AND key = '_skip_import', ft LEFT JOIN node ON node.node_id=ft.ft_id"
            "       WHERE ft.ft_id=ft_ft.master_ft_id AND  "
            "       ft_ft.slave_ft_id = paged_ft.ft_id  AND value IS NOT NULL) ft_ft_json_master, "

            "(SELECT array_agg(ft_addr.addr_id)"
            " FROM ft_addr"
            " WHERE ft_addr.ft_id = paged_ft.ft_id AND"
            "       ft_addr.role = " + ROLE_ZIPCODE_FOR_ADDR +
            ") " + ZIPCODE_ADDR_IDS + " "

            "FROM paged_ft "
            "LEFT JOIN ft_source "
                "ON (paged_ft.ft_id = ft_source.ft_id) "
            "LEFT JOIN source_type "
                "ON (source_type.source_type_id = ft_source.source_type_id) ";
    }

    void tupleToJson(
        json::ObjectBuilder& builder,
        const pqxx::row& tuple) const override
    {
        FtData ft(tuple);
        FtData parentFt(tuple, PARENT_PREFIX);

        builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
            AttributeDumper ad(ft.category(), builder);
            ad.dumpCategory();

            if (ft.hasFtTypeAttr()) {
                ad.dump(FT_TYPE_ID, ft.ftType());
            }

            ad.dump<int>(DISP_CLASS, tuple);

            const auto source_id = tuple.at(SOURCE_ID).as<std::string>(std::string());
            const auto source_name = tuple.at(NAME).as<std::string>(std::string());

            if (source_name == IMPORT_SOURCE_ORG && ft.category().find(POI_PREFIX) == 0) {
                ad.dump(POI_ATTR_PREFIX, BUSINESS_ID, source_id);
            } else {
                ad.dump(category::SYS, IMPORT_SOURCE_ID, source_id);
                ad.dump(category::SYS, IMPORT_SOURCE, source_name);
            }
            if (!tuple[FT_ATTR_JSON].is_null()) {
                auto attrs = maps::json::Value::fromString(tuple[FT_ATTR_JSON].as<std::string>());
                for (const auto& attrName : attrs.fields()) {
                    ad.dump(attrName, attrs[attrName].as<std::string>());
                }
            }
        };

        // If feature has center, ignore polygon
        dumpOptionalGeometry(builder, tuple, POINT_SHAPE)
        || dumpOptionalGeometry(builder, tuple, POLYGON_SHAPE)
        || dumpOptionalGeometry(builder, tuple, LINE_SHAPE);

        if (!parentFt.isNull() ||
            !tuple[FT_FT_JSON_SLAVE].is_null() ||
            !tuple[FT_FT_JSON_MASTER].is_null() ||
            !tuple[ZIPCODE_ADDR_IDS].is_null())
        {
            builder[jkey::RELATIONS] = [&](json::ArrayBuilder builder) {
                RelationDumper rd(builder, tuple);

                if (!parentFt.isNull()) {
                    rd.dump(ft.relation(parentFt), parentFt.id());
                }
                auto relativeCategory = [](const maps::json::Value& relation) {
                    return fixCategory(category::FT,
                                boost::lexical_cast<int>(relation[FT_FT_JSON_FT_TYPE_ID].as<std::string>()),
                                relation[FT_FT_JSON_NOT_POINT].as<std::string>() == FT_FT_JSON_FALSE
                                    ? IsPoint::Yes
                                    : IsPoint::No);
                };
                if (!tuple[FT_FT_JSON_MASTER].is_null()) {
                    std::cerr << tuple[FT_FT_JSON_MASTER].as<std::string>() << std::endl;
                    std::string mastersJson = fixEmbededJSON(tuple[FT_FT_JSON_MASTER].as<std::string>());
                    auto masters =  maps::json::Value::fromString(mastersJson);
                    for (const auto& masterId : masters.fields()) {
                        const auto& relation = masters[masterId];
                        Relation rel {
                            relativeCategory(relation),
                            relation[FT_FT_JSON_ROLE].as<std::string>(),
                            ft.category(),
                            table::FT,
                            RelationDirection::ToMaster
                        };
                        rd.dump(rel, boost::lexical_cast<DBID>(masterId));
                    }
                }
                if (!tuple[FT_FT_JSON_SLAVE].is_null()) {
                    std::string slavesJson = fixEmbededJSON(tuple[FT_FT_JSON_SLAVE].as<std::string>());
                    auto slaves =  maps::json::Value::fromString(slavesJson);
                    for (const auto& slaveId : slaves.fields()) {
                        const auto& relation = slaves[slaveId];
                        Relation rel {
                            ft.category(),
                            relation[FT_FT_JSON_ROLE].as<std::string>(),
                            relativeCategory(relation),
                            table::FT,
                            RelationDirection::ToSlave
                        };
                        rd.dump(rel, boost::lexical_cast<DBID>(slaveId));
                    }
                }
                if (!tuple[ZIPCODE_ADDR_IDS].is_null()) {
                    rd.dump(relation::ADDR_ASSOCIATED_WITH_ZIPCODE, ZIPCODE_ADDR_IDS);
                }
            };
        }
    }

};

DEFINE_CATEGORY_OBJECT(Ft, FT);

} // namespace maps
