#include "../categories.h"
#include "../magic_strings.h"
#include "../attribute_dumper.h"
#include "../relation_dumper.h"

#include <maps/libs/json/include/value.h>
#include <yandex/maps/wiki/common/rd/fow.h>
#include <yandex/maps/wiki/common/rd/lane.h>
#include <yandex/maps/wiki/common/rd/speed_cat.h>

namespace maps {

namespace {

const std::string RD_EL_ID = "rd_el_id";
const std::string F_RD_JC_ID = "f_rd_jc_id";
const std::string T_RD_JC_ID = "t_rd_jc_id";
const std::string FC = "fc";
const std::string FOW = "fow";
const std::string SPEED_CAT = "speed_cat";
const std::string SPEED_LIMIT = "speed_limit";
const std::string F_ZLEV = "f_zlev";
const std::string T_ZLEV = "t_zlev";
const std::string ONEWAY = "oneway";
const std::string ACCESS_ID = "access_id";
const std::string BACK_BUS = "back_bus";
const std::string PAVED = "paved";
const std::string STAIRS = "stairs";
const std::string STRUCT_TYPE = "struct_type";
const std::string FERRY = "ferry";
const std::string DR = "dr";
const std::string TOLL = "toll";
const std::string SRV_RA = "srv_ra";
const std::string SRV_UC = "srv_uc";
const std::string ADDR_FIRST = "addr_first";
const std::string ADDR_LAST = "addr_last";
const std::string ADDR_SCHEME = "addr_scheme";

const std::string RIGHT_SIDE = "1";
const std::string LEFT_SIDE = "2";

const std::string ADDR_FIRST_RIGHT = "addr_a_1";
const std::string ADDR_LAST_RIGHT = "addr_b_1";
const std::string ADDR_SCHEME_RIGHT = "addr_scheme_1";
const std::string ADDR_FIRST_LEFT = "addr_a_2";
const std::string ADDR_LAST_LEFT = "addr_b_2";
const std::string ADDR_SCHEME_LEFT = "addr_scheme_2";
const std::string SCHEME_ODD_1 = "odd";
const std::string SCHEME_EVEN_2 = "even";
const std::string SCHEME_MIXED_3 = "mixed";

const std::string LANES = "lanes";

std::string
getAddrScheme(const std::string& schemAttr, const pqxx::row& tuple)
{
    auto scheme = tuple.at(schemAttr).as<short>(0);
    switch (scheme) {
        case 0:
            return std::string();
        case 1:
            return SCHEME_ODD_1;
        case 2:
            return SCHEME_EVEN_2;
        case 3:
            return SCHEME_MIXED_3;
        default:
            throw RuntimeError() << "Unknown addr_scheme: " << scheme;
    }
}

short
getFow(const pqxx::row& tuple)
{
    auto fow = tuple.at(FOW).as<short>();
    if (fow == 1 || fow == 3 || fow == 12 || fow == 13 || fow == 14) {
        return 0;
    }
    if (fow == 16) {
        return 10;
    }
    return fow;
}

short
getStructType(const pqxx::row& tuple)
{
    auto structType = tuple.at(STRUCT_TYPE).as<short>();
    auto ferry = tuple.at(FERRY).as<short>();
    auto stairs = tuple.at(STAIRS).as<bool>();
    if (ferry > 0) {
        REQUIRE (structType == 0, "Ferry type conflicts with struct type");
        return ferry + 2;
    }
    if (stairs) {
        REQUIRE (structType == 0, "Stairs flag conflicts with struct type");
        return 5;
    }
    return structType;
}

short
getSpeedCat(const pqxx::row& tuple)
{
    auto speedCat = tuple.at(SPEED_CAT).as<short>(0);
    auto fow = static_cast<wiki::common::FOW>(getFow(tuple));
    auto fc = tuple.at(FC).as<short>();
    auto paved = tuple.at(PAVED).as<bool>();

    return speedCat == wiki::common::predictSpeedCategory(fc, fow, paved) ? 0 : speedCat;
}

struct Lanes
{
    std::string tLanes;
    std::string fLanes;
};

typedef std::map<size_t, wiki::common::Lane> OrderedLanes;

std::string
packLanes(const OrderedLanes& lanes)
{
    std::vector<wiki::common::Lane> lanesVector;
    lanesVector.reserve(lanes.size());
    for (const auto& lane : lanes) {
        lanesVector.emplace_back(lane.second);
    }
    return wiki::common::toString(lanesVector);
}

Lanes
decodeLanes(const pqxx::row& tuple)
{
    auto lanesJsonString = tuple.at(LANES).as<std::string>(std::string());
    if (lanesJsonString.empty()) {
        return Lanes {std::string(), std::string()};
    }

    auto json = json::Value::fromString(lanesJsonString);
    ASSERT(json.isArray());
    OrderedLanes tLanes;
    OrderedLanes fLanes;
    for (const auto& lane : json) {
        wiki::common::Lane parsedLane(
            lane[wiki::common::STR_LANE_DIRECTION_ID].as<size_t>(),
            static_cast<wiki::common::LaneKind>(lane[wiki::common::STR_LANE_KIND].as<size_t>()));
        size_t laneNum = lane[wiki::common::STR_LANE_NUM].as<size_t>();
        if (lane[wiki::common::STR_RD_EL_DIRECTION].as<std::string>() == wiki::common::STR_F_RD_EL_DIR) {
            fLanes.emplace(laneNum, parsedLane);
        } else {
            ASSERT(lane[wiki::common::STR_RD_EL_DIRECTION].as<std::string>() == wiki::common::STR_T_RD_EL_DIR);
            tLanes.emplace(laneNum, parsedLane);
        }
    }
    return Lanes {packLanes(tLanes), packLanes(fLanes)};
}

} // namespace

struct RdElCategory: public Category {
    const Table& table() const override
    {
        return table::RD_EL;
    }

    std::string loadRowsSqlTemplate() const override
    {
        return "SELECT "
            "rd_el." + RD_EL_ID + " " + ID + ", " +
            "rd_el." + F_RD_JC_ID + ", " +
            "rd_el." + T_RD_JC_ID + ", " +
            "rd_el." + FC + ", " +
            "rd_el." + FOW + ", " +
            "rd_el." + SPEED_CAT + ", " +
            "rd_el." + SPEED_LIMIT + ", " +
            "rd_el." + F_ZLEV + ", " +
            "rd_el." + T_ZLEV + ", " +
            "rd_el." + ONEWAY + ", " +
            "rd_el." + ACCESS_ID + ", " +
            "rd_el." + BACK_BUS + ", " +
            "rd_el." + PAVED + ", " +
            "rd_el." + STAIRS + ", " +
            "rd_el." + STRUCT_TYPE + ", " +
            "rd_el." + FERRY + ", " +
            "rd_el." + DR + ", " +
            "rd_el." + TOLL + ", " +
            "rd_el." + SRV_RA + ", " +
            "rd_el." + SRV_UC + ", " +
            "ST_AsGeoJSON(rd_el.shape) " + SHAPE + ", "
            "right_range." + ADDR_FIRST + " " + ADDR_FIRST_RIGHT + ", "
            "right_range." + ADDR_LAST + " " + ADDR_LAST_RIGHT + ", "
            "right_range." + ADDR_SCHEME + " " + ADDR_SCHEME_RIGHT + ", "
            "left_range." + ADDR_FIRST + " " + ADDR_FIRST_LEFT + ", "
            "left_range." + ADDR_LAST + " " + ADDR_LAST_LEFT + ", "
            "left_range." + ADDR_SCHEME + " " + ADDR_SCHEME_LEFT + ", "
            "l.lanes " +
            "FROM rd_el "
            "LEFT JOIN addr_range right_range "
                "ON rd_el.rd_el_id=right_range.rd_el_id "
                "AND right_range.addr_side=" + RIGHT_SIDE + " "
            "LEFT JOIN addr_range left_range "
                "ON rd_el.rd_el_id=left_range.rd_el_id "
                "AND left_range.addr_side=" + LEFT_SIDE + " "
            ", LATERAL ("
            "   SELECT array_to_json(array_agg(lane)) lanes"
            "   FROM rd_el_lane lane WHERE lane.rd_el_id = rd_el.rd_el_id) l "
            "WHERE rd_el.rd_el_id in %1% ";
    }

    void tupleToJson(
        json::ObjectBuilder& builder,
        const pqxx::row& tuple) const override
    {
        builder[jkey::ATTRIBUTES] = [&](json::ObjectBuilder builder) {
            AttributeDumper ad(name(), builder);
            ad.dumpCategory();

            ad.dump<int>(FC, tuple);
            ad.dump(SPEED_CAT, getSpeedCat(tuple));
            ad.dump<int>(SPEED_LIMIT, tuple);
            ad.dump<int>(F_ZLEV, tuple);
            ad.dump<int>(T_ZLEV, tuple);
            ad.dump<std::string>(ONEWAY, tuple);
            ad.dump<int>(ACCESS_ID, tuple);
            ad.dump<bool>(BACK_BUS, tuple);
            ad.dump<bool>(PAVED, tuple);
            ad.dump<bool>(DR, tuple);
            ad.dump<bool>(SRV_RA, tuple);
            ad.dump<bool>(SRV_UC, tuple);
            ad.dump(FOW, getFow(tuple));
            ad.dump(STRUCT_TYPE, getStructType(tuple));
            ad.dump<bool>(TOLL, tuple);

            auto lanes = decodeLanes(tuple);
            ad.dump(wiki::common::STR_T_LANE, lanes.tLanes);
            ad.dump(wiki::common::STR_F_LANE, lanes.fLanes);

            if (ad.dump(ADDR_SCHEME_RIGHT, getAddrScheme(ADDR_SCHEME_RIGHT, tuple))) {
                ad.dump<int>(ADDR_FIRST_RIGHT, tuple);
                ad.dump<int>(ADDR_LAST_RIGHT, tuple);
            }

            if (ad.dump(ADDR_SCHEME_LEFT, getAddrScheme(ADDR_SCHEME_LEFT, tuple))) {
                ad.dump<int>(ADDR_FIRST_LEFT, tuple);
                ad.dump<int>(ADDR_LAST_LEFT, tuple);
            }
        };

        dumpGeometry(builder, tuple);

        builder[jkey::RELATIONS] = [&](json::ArrayBuilder builder) {
            RelationDumper rd(builder, tuple);
            rd.dump(relation::RD_EL_START, F_RD_JC_ID);
            rd.dump(relation::RD_EL_END, T_RD_JC_ID);
        };
    }
};

DEFINE_CATEGORY_OBJECT(RdEl, RD_EL);

} // namespace maps
