#include "ymapsdf2json.h"
#include "categories.h"
#include "common.h"
#include "id_manager.h"
#include "file_writer.h"
#include "tasks.h"

#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/pgpool/include/pool_configuration.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/filesystem.hpp>

#include <list>
#include <memory>
#include <iostream>
#include <fstream>

namespace mw = maps::wiki;
namespace bf = boost::filesystem;

using namespace std::chrono_literals;

namespace maps {

namespace {

const std::string MAX_ID_FILE_NAME = "maxid.txt";
const auto CONN_POOL_TIMEOUT_MS = 0s;
const auto CONN_POOL_PING_TIMEOUT_MS = 30s;
const auto CONN_POOL_PING_INTERVAL_MS = 5s;

}

pgpool3::Pool
createPgPool(const std::string& connStr, const PoolSizes& poolSizes)
{
    pgpool3::PoolConstants poolConstants(
        1, 1,
        poolSizes.connPoolSize,
        poolSizes.connPoolSize + poolSizes.connPoolOverflow);

    poolConstants.getTimeoutMs = CONN_POOL_TIMEOUT_MS;
    poolConstants.pingIntervalMs = CONN_POOL_PING_INTERVAL_MS;
    poolConstants.pingTimeoutMs = CONN_POOL_PING_TIMEOUT_MS;

    return pgpool3::Pool(
        connStr,
        poolConstants
    );
}

void run(const Params& params)
{
    PoolSizes poolSizes(params.db.maxConnections);
    auto pgPool = createPgPool(params.db.connString, poolSizes);

    mw::ThreadPool loadIdsPool(poolSizes.loadIdsPoolSize);
    mw::ThreadPool loadRowsPool(poolSizes.loadRowsPoolSize);
    mw::ThreadPool fileWritePool(poolSizes.fileWritePoolSize);

    std::unique_ptr<JsonFileAttributes> jsonFileAttributes;
    {
        auto tr = pgPool.slaveTransaction();
        setSearchPath(*tr, params.db.schema);
        jsonFileAttributes.reset(new JsonFileAttributes(params, *tr));
    }

    initIdManager(params);

    TaskContext context{
        params,
        pgPool,
        loadIdsPool,
        loadRowsPool,
        fileWritePool,
        Writer::make(params),
        *jsonFileAttributes,
        false
    };

    for (const Category& category: CATEGORIES) {
        if (params.categories.empty() || params.categories.count(category.name())) {
            loadIdsPool.push(makeTask<LoadIdsTask>(context, category));
        }
    }

    loadIdsPool.shutdown();
    INFO() << "loadIds finished";

    RegisterIdsTask registerIdsTask(context);
    registerIdsTask.run();
    INFO() << "registerIds finished";

    loadRowsPool.shutdown();
    INFO() << "loadRows finished";

    context.writer->finish(context);
    INFO() << "fileWriter finished";

    fileWritePool.shutdown();
    if (context.failed) {
        throw RuntimeError("One of the tasks failed - see stderr for details");
    }

    if (params.isSplit() || params.idMode == IdMode::Renumber) {
        bf::path path = bf::path(params.outputDir) / MAX_ID_FILE_NAME;
        std::ofstream stream(path.string());
        stream << maxId();
    }
}

}
