from sqlalchemy import create_engine
import yt.wrapper as yt
import argparse
import datetime
import os

from maps.wikimap.stat.libs.common.lib import (
    geobase_region as gb,
    yt_helpers,
)

from maps.wikimap.stat.arm_dump_to_yt.actions import dump_actions
from maps.wikimap.stat.arm_dump_to_yt.objects import dump_objects
from maps.wikimap.stat.arm_dump_to_yt.logger import logger


def main():
    parser = argparse.ArgumentParser(
        description="Use environment variables 'YT_TOKEN' and 'PGPASSWORD' to pass secrets."
    )

    parser.add_argument(
        '--yt-result-folder',
        help='Folder to write dump tables',
        required=True
    )
    parser.add_argument(
        '--db-connection-string',
        help="""Arm database connection string,
        format: postgresql://<user>:@<host>:6432/<db_name>""",
        required=True
    )
    parser.add_argument(
        '--store-time-minutes',
        help='How long we want to store old tables in minutes',
        required=True,
        type=int
    )
    parser.add_argument(
        '--local-run',
        help='Expects to find geobase file "geodata6.bin" in the current directory.',
        action='store_true'
    )

    args = parser.parse_args()

    db_engine = create_engine(args.db_connection_string)

    yt_client = yt.YtClient(
        "hahn",
        token=os.environ['YT_TOKEN']
    )

    if args.local_run:
        gb.init_geodata_from_resource()
    else:
        gb.init_geodata_from_yt_copy(yt_client)

    yesterday = datetime.datetime.utcnow() - datetime.timedelta(days=1)

    with yt.Transaction(client=yt_client):
        dump_actions(yt_client, db_engine, args.yt_result_folder, yesterday)
        dump_objects(yt_client, db_engine, args.yt_result_folder, yesterday)

    yt_helpers.clear_old_tables(
        yt_client,
        args.yt_result_folder,
        args.store_time_minutes,
        logger
    )


if __name__ == '__main__':
    main()
