from maps.wikimap.stat.assessment.graded_units_log.lib import commits, feedback, tracker
import nile.api.v1.utils as nu
from nile.api.v1.stream import BatchStream
from qb2.api.v1 import (
    filters as qf,
    typing,
)


_REDUCER_SCHEMA={
    'unit_id':       typing.Int64,
    'entity_domain': typing.Unicode,
    'entity_id':     typing.Unicode,
    'action_by':     typing.Int64,
    'action_at':     typing.Unicode,
    'action':        typing.Unicode,
    'value':         typing.Unicode,
    'certainty':     typing.Unicode,
}
GRADED_UNITS_LOG_SCHEMA = {
    **_REDUCER_SCHEMA,
    'region_id': typing.Int64,
    'task_id':   typing.Unicode,
}


class LatestGrade(object):
    def __init__(self, qualification):
        self._qualification = qualification
        self._graded_at = None
        self._value = None

    def add_grade(self, graded_at, value):
        if self._graded_at is None or self._graded_at < graded_at:
            self._value = value
            self._graded_at = graded_at

    def get(self):
        if self._value is not None:
            return {
                'value':     self._value,
                'certainty': self._qualification,
            }


class GradesConsensus(object):
    def __init__(self, qualification):
        self._qualification = qualification
        self._balance = 0
        self._count = 0

    def add_grade(self, value):
        if value == 'correct':
            self._balance += 1
        elif value == 'incorrect':
            self._balance -= 1
        else:
            raise RuntimeError(f'Unexpected grade value {value}')
        self._count += 1

    def _get_value(self):
        if self._balance != 0:
            return 'correct' if self._balance > 0 else 'incorrect'
        return 'conflict'

    def _get_certainty(self):
        if self._count == 1:
            return 'single'
        elif self._count > abs(self._balance):
            return 'majority'
        elif self._count > 0:
            return 'unanimous'

    def get(self):
        if self._count > 0:
            return {
                'value':     self._get_value(),
                'certainty': self._qualification + '_' + self._get_certainty(),
            }


class GradesFinalValue(object):
    def __init__(self):
        self._expert = LatestGrade('expert')
        self._basic = GradesConsensus('basic')
        self._arbitrary = LatestGrade('arbitrary')

    def add_grade(self, graded_at, value, qualification):
        if qualification == 'expert':
            self._expert.add_grade(graded_at, value)
        elif qualification == 'basic':
            self._basic.add_grade(value)
        elif qualification is None:
            self._arbitrary.add_grade(graded_at, value)
        else:
            raise RuntimeError(f'Unexpected grade qualification {qualification}')

    def get(self):
        return (
            self._expert.get() or
            self._basic.get() or
            self._arbitrary.get()
        )


@nu.with_hints(output_schema=_REDUCER_SCHEMA)
def _graded_units_reducer(groups):
    for key, records in groups:
        grades_final_value = GradesFinalValue()
        for record in records:
            grades_final_value.add_grade(
                record['graded_at'],
                record['value'].decode(),
                record.get('qualification'),
            )

        result = grades_final_value.get()
        if result is not None:
            yield record.transform('grade_id', 'qualification', **result)


def _get_graded_units(unit_table, grade_table, min_date):
    '''
    unit_table:
    | unit_id | entity_id | entity_domain | action_by | action_at | action |
    |---------+-----------+---------------+-----------+-----------+--------|
    | ...     | ...       | ...           | ...       | ...       | ...    |

    grade_table:
    | unit_id | graded_at | value | qualification | ... |
    |---------+-----------+-------+---------------+-----|
    | ...     | ...       | ...   | ...           | ... |

    result:
    | unit_id | entity_id | entity_domain | action_by | action_at | action | value | qualification |
    |---------+-----------+---------------+-----------+-----------+--------+-------+--------|
    | ...     | ...       | ...           | ...       | ...       | ...    | ...   | ...    |
    '''
    return grade_table.filter(
        qf.compare('graded_at', '>=', min_date),
    ).project(
        'unit_id', 'graded_at', 'value', 'qualification',
    ).join(
        unit_table.filter(qf.compare('action_at', '>=', min_date)),
        by='unit_id',
        assume_small_left=True,
        assume_small_right=True,
        assume_unique_right=True,
    ).groupby(
        'unit_id'
    ).reduce(
        _graded_units_reducer
    )


def _split_by_domains(graded_units_log, *domains):
    return BatchStream(graded_units_log.split(
        *[qf.equals('entity_domain', domain) for domain in domains],
        multisplit=True,
        strategy='stop_if_true',
    )[:-1])


def make_graded_units_log(
    job,
    dates,
    grade_table,
    unit_table,
    commit_event_table,
    social_task_table,
    feedback_task_table,
    components_table,
    issue_events_table,
    issues_table,
    resolutions_table,
    statuses_table,
):
    '''
    grade_table:
    | grade_id | unit_id | value | qualification | ... |
    |----------+---------+-------+---------------+-----|
    | ...      | ...     | ...   | ...           | ... |

    unit_table:
    | unit_id | entity_id | entity_domain | action_by | action_at | action |
    |---------+-----------+---------------+-----------+-----------+--------|
    | ...     | ...       | ...           | ...       | ...       | ...    |

    social_task_table:
    | event_id | resolved_at | resolved_by | closed_at | closed_by | primary_object_category_id | ... |
    |----------+-------------+-------------+-----------+-----------+----------------------------+-----|
    | ...      | ...         | ...         | ...       | ...       | ...                        | ... |

    commit_event_table:
    | commit_id | event_id | created_at | bounds_geom | primary_object_category_id | ... |
    |-----------+----------+------------+-------------+----------------------------+-----|
    | ...       | ...      | ...        | ...         | ...                        | ... |

    feedback_task_table:
    | id  | created_at | position | qualification | type | ... |
    |-----+------------+----------+---------------+------+-----|
    | ... | ...        | ...      | ...           | ...  | ... |

    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    issue_events_table:
    | issue | date (ms) | changes (yson)             | ... |
    |-------+-----------+----------------------------+-----|
    | ...   | ...       | [                          | ... |
    |       |           |   {                        |     |
    |       |           |      'field': 'fieldName', |     |
    |       |           |      'newValue': {         |     |
    |       |           |        'value': yson_value |     |
    |       |           |      },                    |     |
    |       |           |      ...                   |     |
    |       |           |   },                       |     |
    |       |           |   ...                      |     |
    |       |           | ]                          |     |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | ... |
    |-----+-----------------------+------------+----------+------------+-------------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ... |

    resolutions_table:
    | id  | key                           | ... |
    |-----+-------------------------------+-----|
    | ... | internal name of a resolution | ... |

    statuses_table:
    | id  | key                       | ... |
    |-----+---------------------------+-----|
    | ... | internal name of a status | ... |

    result:
    | unit_id | entity_id | entity_domain | action_by | action_at | action | value | qualification | region_id | task_id |
    |---------+-----------+---------------+-----------+-----------+--------+-------+---------------+-----------+---------|
    | ...     | ...       | ...           | ...       | ...       | ...    | ...   | ...           | ...       | ...     |
    '''
    edits_units, moderation_units, feedback_units, tracker_units = _split_by_domains(
        _get_graded_units(unit_table, grade_table, min(dates)),
        'edits', 'moderation', 'feedback', 'tracker',
    )

    return job.concat(
        commits.add_task_id_region_id(
            edits_units,
            commits.get_edits_events(commit_event_table, min(dates)),
        ),
        commits.add_task_id_region_id(
            moderation_units,
            commits.get_moderation_tasks(social_task_table, commit_event_table, min(dates)),
        ),
        feedback.add_task_id_region_id(
            feedback_units,
            feedback.get_feedback_tasks(feedback_task_table, min(dates)),
        ),
        tracker.add_task_id_region_id(
            tracker_units,
            tracker.get_tracker_issues(
                dates,
                components_table=components_table,
                issue_events_table=issue_events_table,
                issues_table=issues_table,
                resolutions_table=resolutions_table,
                statuses_table=statuses_table,
            )
        )
    ).cast(**GRADED_UNITS_LOG_SCHEMA)
