from maps.wikimap.stat.libs.tracker import (
    filter_by_queues,
    get_closed_issues,
    get_tracker_task_id,
)
from maps.wikimap.stat.libs.geobase_regions import EARTH_REGION_ID
import nile.api.v1.extractors as ne
from qb2.api.v1 import typing

GRADED_QUEUES = frozenset({'GEOCONTENTFB'})


def get_tracker_issues(
    dates,
    components_table,
    issue_events_table,
    issues_table,
    resolutions_table,
    statuses_table,
):
    '''
    components_table:
    | id  | name | ... |
    |-----+------+-----|
    | ... | ...  | ... |

    issue_events_table:
    | issue | date (ms) | changes (yson)             | ... |
    |-------+-----------+----------------------------+-----|
    | ...   | ...       | [                          | ... |
    |       |           |   {                        |     |
    |       |           |      'field': 'fieldName', |     |
    |       |           |      'newValue': {         |     |
    |       |           |        'value': yson_value |     |
    |       |           |      },                    |     |
    |       |           |      ...                   |     |
    |       |           |   },                       |     |
    |       |           |   ...                      |     |
    |       |           | ]                          |     |

    issues_table:
    | id  | key                   | resolution | assignee | components | storyPoints | ... |
    |-----+-----------------------+------------+----------+------------+-------------+-----|
    | ... | <queue name>-<number> | ...        | ...      | ...        | ...         | ... |

    resolutions_table:
    | id  | key                           | ... |
    |-----+-------------------------------+-----|
    | ... | internal name of a resolution | ... |

    statuses_table:
    | id  | key                       | ... |
    |-----+---------------------------+-----|
    | ... | internal name of a status | ... |

    result:
    | entity_id             | component |
    |-----------------------+-----------|
    | GEOCONTENTFB-<number> | ...       |
    '''
    return get_closed_issues(
        dates,
        components_table,
        issue_events_table,
        filter_by_queues(issues_table, GRADED_QUEUES),
        resolutions_table,
        statuses_table
    ).project(
        'component',
        entity_id='ticket',
    ).cast(
        entity_id=typing.Unicode,
    )


def add_task_id_region_id(tracker_units, tracker_issues):
    '''
    tracker_units:
    | entity_id | entity_domain | ... |
    |-----------+---------------+-----|
    | ...       | "tracker"     | ... |

    tracker_issues:
    | entity_id | component |
    |-----------+-----------|
    | ...       | ...       |

    result:
    | entity_id | entity_domain | task_id | region_id       | ... |
    |-----------+---------------+---------+-----------------+-----|
    | ...       | "tracker"     | ...     | EARTH_REGION_ID | ... |
    '''
    return tracker_units.join(
        tracker_issues,
        by='entity_id',
        assume_small_right=True,
    ).project(
        ne.all(exclude='component'),
        task_id=ne.custom(get_tracker_task_id, 'entity_id', 'component').with_type(typing.Unicode),
        region_id=ne.const(EARTH_REGION_ID).with_type(typing.Int64),
    )
